# Localization of Git LFS

Git LFS now has support for a localization framework using [Gotext](https://github.com/leonelquinteros/gotext), a Go library based around the popular gettext format.
Localization is important since the majority of people on the planet don't speak English and people should be able to use software in the language they're most comfortable with.
In addition, having access to localized software is a great way for people learning a language to improve their technical vocabulary.

## Choosing What to Translate

Here are some things that should be translated:

* Status messages
* Error messages
* Help output
* Generally, anything that the user sees in the normal course of operation

Here are some things that should not be translated:

* Trace output (e.g., calls to `tracerx.Printf`)
* Strings which have a functional or protocol use (e.g., `Basic` and other authentication schemes, HTTP verbs)
* Names of programs, commands, command-line options, or subcommands (although you _should_ translate their help output)
* Personal names, names of businesses, email addresses, and other proper nouns (most of which we should not typically have in our codebase)
* The names of Git LFS and Git themselves

## Making Text Translatable

The easiest way to make a string translatable is to wrap the string and any formatting arguments it takes in a call to `tr.Tr.Get`.
For example, you might write this:

```
Print(tr.Tr.Get("fetch: Fetching reference %s", ref.Name))
```

If you have a string which varies based on a number, use `tr.Tr.GetN`, provide the singular string, the plural string, the number upon which it varies, and then the arguments:

```
Print(tr.Tr.GetN(
	"fetch: Fetching changes within %v day of %v",
	"fetch: Fetching changes within %v days of %v",
	fetchconf.FetchRecentCommitsDays,
	fetchconf.FetchRecentCommitsDays,
	refName,
))
```

Here are some tips for making your code as easy to translate as possible:

* Avoid creating strings out of several independent words.
  For example, rather than taking a string like "Upload" or "Download" and appending "ing objects" to it, write the strings out in full as "Uploading objects" and "Downloading objects".
  Not all languages compose words and phrases in the same way, and using full sentences or phrases makes it easier to translate.
* If you have a string which varies based on number, use `tr.Tr.GetN`, which handles pluralization correctly.
  In some languages, zero is singular instead of plural.  Also, unlike English, some languages have more than just two forms of a word (e.g., singular, dual, and plural), or words may not vary in number at all.
  This also lets users in English see correctly pluralized strings in all cases.
* Only mark literal strings for translation, and mark them wherever they appear in the codebase.
  The `xgotext` binary rips out only literal strings and doesn't handle variables, so strings that are not marked for translation at the same place they appear in the text won't end up in the translation files to be translated.
* Try to use text in strings that is simple and direct, avoiding the use of slang, idioms, or regional varieties of English.
  Slang, idioms, and regional vocabulary are hard to translate and they aren't as accessible to people who are learning English as a second language.

## Guidelines for Translators

If you're interested in submitting a translation, please open an issue, and we'll work with you to get what you need.
We'll let you know a little in advance of our next planned release to give you time to update the translations.

When choosing to create a translation, we ask that you write a generic translation when possible (e.g., `es` rather than `es_MX`).
We realize that is not always achievable (such as with `pt_BR` and `pt_PT`), but when possible, it makes the translations more accessible.

For vocabulary, we recommend using the same rules as Git whenever possible.
If there's a decision to be made, picking the more universally intelligible option (e.g., _quatre-vingts_ instead of _huitante_ in French or _ustedes_ instead of _vosotros_ in Spanish) is preferable.
We'll generally defer to your experience on these matters.

To extract the strings for translation, run the following commands:

```
$ go install -v github.com/leonelquinteros/gotext/cli/xgotext
$ xgotext -in . -out po -v
```

Be aware that at the moment, `xgotext` is extremely slow.
You may wish to get a coffee while you wait.

## Help

If you're unclear on the best way to do things or you find a problem, feel free to open an issue or discussion, as appropriate, and we'll be happy to work with you to find a solution.
