//byteorder.h:

/*
 *      Copyright (C) Philipp 'ph3-der-loewe' Schafft - 2010-2013
 *
 *  This file is part of aroarfw, a RoarAudio framework for
 *  embedded systems (Controlers).
 *
 *  This file is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 3
 *  or (at your option) any later version as published by
 *  the Free Software Foundation.
 *
 *  aroarfw is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this software; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 51 Franklin Street, Fifth Floor,
 *  Boston, MA 02110-1301, USA.
 */

#ifndef _AROARFW_BYTEORDER_H_
#define _AROARFW_BYTEORDER_H_

/** \brief Convert a 16 bit integer into a 2 byte buffer in network byte order.

 This function converts a 16 bit integer in host byte order
 into network byte order and puts it into a 2 byte buffer.

 \param buf The buffer to put the integer in
 \param v   The integer to put in the buffer
 \return The integer in host byte order
*/
#define RBO_H2NB16(buf, v) ( \
                            ((uint16_t)(((unsigned char*)(buf))[0] = ((uint16_t)(v) >> 8) & 0xFF) << 8) + \
                             (uint16_t)(((unsigned char*)(buf))[1] =  (uint16_t)(v)       & 0xFF) \
                           )

/** \brief Convert a 2 byte buffer into a 16 bit integer in host byte order.

 This function converts a 2 byte buffer in network byte order into a
 16 bit integer in host byte order.

 \param buf The buffer to get the integer from
 \return The integer in host byte order
*/
#define RBO_NB2H16(buf)    ( \
                            ((uint16_t)(((unsigned char*)(buf))[0]) << 8) + \
                             (uint16_t)(((unsigned char*)(buf))[1]) \
                           )

/** \brief Convert a 32 bit integer into a 4 byte buffer in network byte order.

 This function converts a 32 bit integer in host byte order
 into network byte order and puts it into a 4 byte buffer.

 \param buf The buffer to put the integer in
 \param v   The integer to put in the buffer
 \return The integer in host byte order
*/
#define RBO_H2NB32(buf, v) ( \
                            ((uint32_t)(((unsigned char*)(buf))[0] = ((uint32_t)(v) >> 24) & 0xFF) << 24) + \
                            ((uint32_t)(((unsigned char*)(buf))[1] = ((uint32_t)(v) >> 16) & 0xFF) << 16) + \
                            ((uint32_t)(((unsigned char*)(buf))[2] = ((uint32_t)(v) >>  8) & 0xFF) <<  8) + \
                             (uint32_t)(((unsigned char*)(buf))[3] =  (uint32_t)(v)        & 0xFF) \
                           )

/** \brief Convert a 4 byte buffer into a 32 bit integer in host byte order.

 This function converts a 4 byte buffer in network byte order into a
 32 bit integer in host byte order.

 \param buf The buffer to get the integer from
 \return The integer in host byte order
*/
#define RBO_NB2H32(buf)    ( \
                            ((uint32_t)(((unsigned char*)(buf))[0]) << 24) + \
                            ((uint32_t)(((unsigned char*)(buf))[1]) << 16) + \
                            ((uint32_t)(((unsigned char*)(buf))[2]) <<  8) + \
                             (uint32_t)(((unsigned char*)(buf))[3]) \
                           )

#endif

//ll
