/*****************************************************************************
* $CAMITK_LICENCE_BEGIN$
*
* CamiTK - Computer Assisted Medical Intervention ToolKit
* (c) 2001-2024 Univ. Grenoble Alpes, CNRS, Grenoble INP - UGA, TIMC, 38000 Grenoble, France
*
* Visit http://camitk.imag.fr for more information
*
* This file is part of CamiTK.
*
* CamiTK is free software: you can redistribute it and/or modify
* it under the terms of the GNU Lesser General Public License version 3
* only, as published by the Free Software Foundation.
*
* CamiTK is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU Lesser General Public License version 3 for more details.
*
* You should have received a copy of the GNU Lesser General Public License
* version 3 along with CamiTK.  If not, see <http://www.gnu.org/licenses/>.
*
* $CAMITK_LICENCE_END$
****************************************************************************/

#include "BitmapViewer.h"

#include <Application.h>
#include <MainWindow.h>
#include <SingleImageComponent.h>
#include <MedicalImageViewer.h>
#include <Log.h>

#include <QToolBar>

// disable warning generated by clang about the surrounded headers
#include <CamiTKDisableWarnings>
#include <vtkCamera.h> // to change the camera projection to parallel
#include <CamiTKReEnableWarnings>

using namespace camitk;

// -------------------- Constructor --------------------
BitmapViewer::BitmapViewer(QString name) : InteractiveViewer(name, InteractiveViewer::SLICE_VIEWER) {
    // 2D images need x right, y up, z front
    getRendererWidget()->setCameraOrientation(RendererWidget::RIGHT_DOWN);
    // no decoration and grey background
    getRendererWidget()->toggleOrientationDecorations(false);
    getRendererWidget()->setGradientBackground(false);
    getRendererWidget()->setBackgroundColor(0.8, 0.8, 0.8);
    // parallel projection
    getRendererWidget()->getActiveCamera()->ParallelProjectionOn();

    // can only display 2D bitmap = SingleImageComponent
    setComponentClassNames(QStringList() << "SingleImageComponent");

    setDescription("BitmapViewer is a simple 2D image viewer");

    myWidget = nullptr;
    myToolbar = nullptr;
}

// -------------------- Destructor --------------------
BitmapViewer::~BitmapViewer() {
    // do not delete myWidget as it will automatically be deleted
    // when the embedder widget will be deleted
}

// -------------------- refresh --------------------
void BitmapViewer::refresh(Viewer* whoIsAsking) {
    updateVisibility();
    InteractiveViewer::refresh();
}

// -------------------- getWidget --------------------
QWidget* BitmapViewer::getWidget() {
    if (myWidget == nullptr) {
        myWidget = InteractiveViewer::getWidget();
        // hide slice slider
        sideFrame->hide();
    }
    updateVisibility();
    return myWidget;
}

// -------------------- getToolBar --------------------
QToolBar* BitmapViewer::getToolBar() {
    if (myToolbar == nullptr) {
        // create the toolbar and add the action to it
        myToolbar = new QToolBar(objectName() + " Toolbar");
        // ensure object name is set for saving the state
        myToolbar->setObjectName(objectName() + " Toolbar");

        myToolbar->addAction(screenshotAction);

        MedicalImageViewer* medicalImageViewer = dynamic_cast<MedicalImageViewer*>(Application::getViewer("Medical Image Viewer"));

        // create the action to switch back to the medical image viewer
        QAction* hideViewerAction = myToolbar->addAction(tr("Hide Bitmap Viewer"));
        hideViewerAction->setIcon(QPixmap(":/fileClose"));
        hideViewerAction->setStatusTip(tr("Switch back to the medical image viewer"));
        hideViewerAction->setWhatsThis(tr("Switch back the ventral viewer to normal (i.e., show the Medical Image Viewer)"));
        connect(hideViewerAction, &QAction::triggered, [ = ](bool) {
            Application::getMainWindow()->setCentralViewer(medicalImageViewer);
        });
    }

    return myToolbar;
}


// -------------------- updateVisibility --------------------
void BitmapViewer::updateVisibility() {
    // As the default component does not know anything about BitmapViewer
    // tell all the SingleImageComponent components of axial orientation to be
    // visible in this viewer
    for (Component* comp : Application::getAllComponents()) {
        SingleImageComponent* singleImageComp = dynamic_cast<SingleImageComponent*>(comp);
        if (singleImageComp != nullptr && singleImageComp->getSliceOrientation() == Slice::AXIAL) {
            singleImageComp->setVisibility(this->getName(), true);
        }
    }
}

