/* Internals of libgccjit: classes for playing back recorded API calls.
   Copyright (C) 2013-2026 Free Software Foundation, Inc.
   Contributed by David Malcolm <dmalcolm@redhat.com>.

This file is part of GCC.

GCC is free software; you can redistribute it and/or modify it
under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 3, or (at your option)
any later version.

GCC is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with GCC; see the file COPYING3.  If not see
<http://www.gnu.org/licenses/>.  */

#include "config.h"
#define INCLUDE_MUTEX
#define INCLUDE_DLFCN_H
#include "libgccjit.h"
#include "system.h"
#include "coretypes.h"
#include "target.h"
#include "tree.h"
#include "stringpool.h"
#include "cgraph.h"
#include "dumpfile.h"
#include "toplev.h"
#include "tree-cfg.h"
#include "convert.h"
#include "gimple-expr.h"
#include "stor-layout.h"
#include "print-tree.h"
#include "gimplify.h"
#include "gcc-driver-name.h"
#include "attribs.h"
#include "context.h"
#include "fold-const.h"
#include "opt-suggestions.h"
#include "gcc.h"
#include "diagnostic.h"
#include "stmt.h"
#include "realmpfr.h"

#include "jit-playback.h"
#include "jit-result.h"
#include "jit-builtins.h"
#include "jit-tempdir.h"
#include "jit-target.h"

#ifdef _WIN32
#include "jit-w32.h"
#endif

/* Compare with gcc/c-family/c-common.h: DECL_C_BIT_FIELD,
   SET_DECL_C_BIT_FIELD.
   These are redefined here to avoid depending from the C frontend.  */
#define DECL_JIT_BIT_FIELD(NODE) \
  (DECL_LANG_FLAG_4 (FIELD_DECL_CHECK (NODE)) == 1)
#define SET_DECL_JIT_BIT_FIELD(NODE) \
  (DECL_LANG_FLAG_4 (FIELD_DECL_CHECK (NODE)) = 1)

/* gcc::jit::playback::context::build_cast uses the convert.h API,
   which in turn requires the frontend to provide a "convert"
   function, apparently as a fallback for casts that can be simplified
   (truncation, extension). */
extern tree convert (tree type, tree expr);

tree
convert (tree dst_type, tree expr)
{
  tree t_ret = NULL;
  t_ret = targetm.convert_to_type (dst_type, expr);
  if (t_ret)
      return t_ret;
  switch (TREE_CODE (dst_type))
    {
    case INTEGER_TYPE:
    case ENUMERAL_TYPE:
      return fold (convert_to_integer (dst_type, expr));

    default:
      gcc_assert (gcc::jit::active_playback_ctxt);
      gcc::jit::active_playback_ctxt->add_error (NULL, "unhandled conversion");
      fprintf (stderr, "input expression:\n");
      debug_tree (expr);
      fprintf (stderr, "requested type:\n");
      debug_tree (dst_type);
      return error_mark_node;
    }
}

namespace gcc {
namespace jit {

/**********************************************************************
 Playback.
 **********************************************************************/

/* Fold a readonly non-volatile variable with an initial constant value,
   to that value.

   Otherwise return the argument unchanged.

   This fold is needed for setting a variable's DECL_INITIAL to the value
   of a const variable.  The c-frontend does this in its own special
   fold (), so we lift this part out and do it explicitly where there is a
   potential for variables to be used as rvalues.  */
static tree
fold_const_var (tree node)
{
  /* See c_fully_fold_internal in c-fold.cc and decl_constant_value_1
     in c-typeck.cc.  */
  if (VAR_P (node)
      && TREE_READONLY (node)
      && !TREE_THIS_VOLATILE (node)
      && DECL_INITIAL (node) != NULL_TREE
      /* "This is invalid if initial value is not constant.
	  If it has either a function call, a memory reference,
	  or a variable, then re-evaluating it could give different
	  results."  */
      && TREE_CONSTANT (DECL_INITIAL (node)))
    {
      tree ret = DECL_INITIAL (node);
      /* "Avoid unwanted tree sharing between the initializer and current
	  function's body where the tree can be modified e.g. by the
	  gimplifier."  */
      if (TREE_STATIC (node))
	ret = unshare_expr (ret);

      return ret;
    }

  return node;
}

/* Build a STRING_CST tree for STR, or return NULL if it is NULL.
   The TREE_TYPE is not initialized.  */

static tree
build_string (const char *str)
{
  if (str)
    return ::build_string (strlen (str), str);
  else
    return NULL_TREE;
}

/* The constructor for gcc::jit::playback::context.  */

playback::context::context (recording::context *ctxt)
  : log_user (ctxt->get_logger ()),
    m_recording_ctxt (ctxt),
    m_tempdir (NULL),
    m_const_char_ptr (NULL)
{
  JIT_LOG_SCOPE (get_logger ());
  m_functions.create (0);
  m_globals.create (0);
  m_source_files.create (0);
  m_cached_locations.create (0);
}

/* The destructor for gcc::jit::playback::context.  */

playback::context::~context ()
{
  JIT_LOG_SCOPE (get_logger ());

  /* Normally the playback::context is responsible for cleaning up the
     tempdir (including "fake.so" within the filesystem).

     In the normal case, clean it up now.

     However m_tempdir can be NULL if the context has handed over
     responsibility for the tempdir cleanup to the jit::result object, so
     that the cleanup can be delayed (see PR jit/64206).  If that's the
     case this "delete NULL;" is a no-op. */
  delete m_tempdir;

  m_functions.release ();
}

/* A playback::context can reference GC-managed pointers.  Mark them
   ("by hand", rather than by gengtype).

   This is called on the active playback context (if any) by the
   my_ggc_walker hook in the jit_root_table in dummy-frontend.cc.  */

void
playback::context::
gt_ggc_mx ()
{
  int i;
  function *func;
  FOR_EACH_VEC_ELT (m_functions, i, func)
    {
      if (ggc_test_and_set_mark (func))
	func->gt_ggc_mx ();
    }
}

/* Given an enum gcc_jit_types value, get a "tree" type.  */

tree
playback::context::
get_tree_node_for_type (enum gcc_jit_types type_)
{
  switch (type_)
    {
    case GCC_JIT_TYPE_VOID:
      return void_type_node;

    case GCC_JIT_TYPE_VOID_PTR:
      return ptr_type_node;

    case GCC_JIT_TYPE_BOOL:
      return boolean_type_node;

    case GCC_JIT_TYPE_CHAR:
      return char_type_node;
    case GCC_JIT_TYPE_SIGNED_CHAR:
      return signed_char_type_node;
    case GCC_JIT_TYPE_UNSIGNED_CHAR:
      return unsigned_char_type_node;

    case GCC_JIT_TYPE_SHORT:
      return short_integer_type_node;
    case GCC_JIT_TYPE_UNSIGNED_SHORT:
      return short_unsigned_type_node;

    case GCC_JIT_TYPE_CONST_CHAR_PTR:
      return m_const_char_ptr;

    case GCC_JIT_TYPE_INT:
      return integer_type_node;
    case GCC_JIT_TYPE_UNSIGNED_INT:
      return unsigned_type_node;

    case GCC_JIT_TYPE_UINT8_T:
      return unsigned_intQI_type_node;
    case GCC_JIT_TYPE_UINT16_T:
      return uint16_type_node;
    case GCC_JIT_TYPE_UINT32_T:
      return uint32_type_node;
    case GCC_JIT_TYPE_UINT64_T:
      return uint64_type_node;
    case GCC_JIT_TYPE_UINT128_T:
      if (targetm.scalar_mode_supported_p (TImode))
	return uint128_type_node;

      add_error (NULL, "gcc_jit_types value unsupported on this target: %i",
		 type_);
      return NULL;

    case GCC_JIT_TYPE_INT8_T:
      return intQI_type_node;
    case GCC_JIT_TYPE_INT16_T:
      return intHI_type_node;
    case GCC_JIT_TYPE_INT32_T:
      return intSI_type_node;
    case GCC_JIT_TYPE_INT64_T:
      return intDI_type_node;
    case GCC_JIT_TYPE_INT128_T:
      if (targetm.scalar_mode_supported_p (TImode))
	return intTI_type_node;

      add_error (NULL, "gcc_jit_types value unsupported on this target: %i",
		 type_);
      return NULL;

    case GCC_JIT_TYPE_LONG:
      return long_integer_type_node;
    case GCC_JIT_TYPE_UNSIGNED_LONG:
      return long_unsigned_type_node;

    case GCC_JIT_TYPE_LONG_LONG:
      return long_long_integer_type_node;
    case GCC_JIT_TYPE_UNSIGNED_LONG_LONG:
      return long_long_unsigned_type_node;

    case GCC_JIT_TYPE_FLOAT:
      return float_type_node;
    case GCC_JIT_TYPE_BFLOAT16:
#ifndef HAVE_BFmode
      add_error (NULL, "gcc_jit_types value unsupported on this target: %i",
		 type_);
#endif
      return bfloat16_type_node;
    case GCC_JIT_TYPE_DOUBLE:
      return double_type_node;
    case GCC_JIT_TYPE_LONG_DOUBLE:
      return long_double_type_node;
    case GCC_JIT_TYPE_FLOAT16:
      if (float16_type_node == NULL || TYPE_PRECISION (float16_type_node) != 16)
      {
	add_error (NULL, "gcc_jit_types value unsupported on this target: %i",
		   type_);
	return NULL;
      }
      return float16_type_node;
    case GCC_JIT_TYPE_FLOAT32:
      if (float32_type_node == NULL || TYPE_PRECISION (float32_type_node) != 32)
      {
	add_error (NULL, "gcc_jit_types value unsupported on this target: %i",
		   type_);
	return NULL;
      }
      return float32_type_node;
    case GCC_JIT_TYPE_FLOAT64:
      if (float64_type_node == NULL || TYPE_PRECISION (float64_type_node) != 64)
      {
	add_error (NULL, "gcc_jit_types value unsupported on this target: %i",
		   type_);
	return NULL;
      }
      return float64_type_node;
    case GCC_JIT_TYPE_FLOAT128:
      if (float128_type_node == NULL
	  || TYPE_PRECISION (float128_type_node) != 128)
      {
	add_error (NULL, "gcc_jit_types value unsupported on this target: %i",
		   type_);
	return NULL;
      }
      return float128_type_node;

    case GCC_JIT_TYPE_SIZE_T:
      return size_type_node;

    case GCC_JIT_TYPE_FILE_PTR:
      return fileptr_type_node;

    case GCC_JIT_TYPE_COMPLEX_FLOAT:
      return complex_float_type_node;
    case GCC_JIT_TYPE_COMPLEX_DOUBLE:
      return complex_double_type_node;
    case GCC_JIT_TYPE_COMPLEX_LONG_DOUBLE:
      return complex_long_double_type_node;
    }

  add_error (NULL, "unrecognized (enum gcc_jit_types) value: %i",
	     type_);

  return NULL;
}

/* Construct a playback::type instance (wrapping a tree) for the given
   enum value.  */

playback::type *
playback::context::
get_type (enum gcc_jit_types type_)
{
  tree type_node = get_tree_node_for_type (type_);
  if (type_node == NULL)
    return NULL;

  return new type (type_node);
}

void
playback::context::
set_output_ident (const char* ident)
{
  targetm.asm_out.output_ident (ident);
}

/* Construct a playback::type instance (wrapping a tree) for the given
   array type.  */

playback::type *
playback::context::
new_array_type (playback::location *loc,
		playback::type *element_type,
		uint64_t num_elements)
{
  gcc_assert (element_type);

  tree t = build_array_type_nelts (element_type->as_tree (),
				   num_elements);
  layout_type (t);

  if (loc)
    set_tree_location (t, loc);

  return new type (t);
}

/* Construct a playback::field instance (wrapping a tree).  */

playback::field *
playback::context::
new_field (location *loc,
	   type *type,
	   const char *name)
{
  gcc_assert (type);
  gcc_assert (name);

  /* compare with c/c-decl.cc:grokfield and grokdeclarator.  */
  tree decl = build_decl (UNKNOWN_LOCATION, FIELD_DECL,
			  get_identifier (name), type->as_tree ());

  if (loc)
    set_tree_location (decl, loc);

  return new field (decl);
}

/* Construct a playback::bitfield instance (wrapping a tree).  */

playback::field *
playback::context::
new_bitfield (location *loc,
	      type *type,
	      int width,
	      const char *name)
{
  gcc_assert (type);
  gcc_assert (name);
  gcc_assert (width);

  /* compare with c/c-decl.cc:grokfield,  grokdeclarator and
     check_bitfield_type_and_width.  */

  tree tree_type = type->as_tree ();
  gcc_assert (INTEGRAL_TYPE_P (tree_type));
  tree tree_width = build_int_cst (integer_type_node, width);
  if (compare_tree_int (tree_width, TYPE_PRECISION (tree_type)) > 0)
    {
      add_error (
	loc,
	"width of bit-field %s (width: %i) is wider than its type (width: %i)",
	name, width, TYPE_PRECISION (tree_type));
      return NULL;
    }

  tree decl = build_decl (UNKNOWN_LOCATION, FIELD_DECL,
			  get_identifier (name), type->as_tree ());
  DECL_NONADDRESSABLE_P (decl) = true;
  DECL_INITIAL (decl) = tree_width;
  SET_DECL_JIT_BIT_FIELD (decl);

  if (loc)
    set_tree_location (decl, loc);

  return new field (decl);
}

/* Construct a playback::compound_type instance (wrapping a tree).  */

playback::compound_type *
playback::context::
new_compound_type (location *loc,
		   const char *name,
		   bool is_struct) /* else is union */
{
  gcc_assert (name);

  /* Compare with c/c-decl.cc: start_struct. */

  tree t = make_node (is_struct ? RECORD_TYPE : UNION_TYPE);
  TYPE_NAME (t) = get_identifier (name);
  TYPE_SIZE (t) = 0;

  if (loc)
    set_tree_location (t, loc);

  return new compound_type (t);
}

void
playback::compound_type::set_fields (const auto_vec<playback::field *> *fields)
{
  /* Compare with c/c-decl.cc: finish_struct. */
  tree t = as_tree ();

  tree fieldlist = NULL;
  for (unsigned i = 0; i < fields->length (); i++)
    {
      field *f = (*fields)[i];
      tree x = f->as_tree ();
      DECL_CONTEXT (x) = t;
      if (DECL_JIT_BIT_FIELD (x))
	{
	  unsigned HOST_WIDE_INT width = tree_to_uhwi (DECL_INITIAL (x));
	  DECL_SIZE (x) = bitsize_int (width);
	  DECL_BIT_FIELD (x) = 1;
	}
      fieldlist = chainon (x, fieldlist);
    }
  fieldlist = nreverse (fieldlist);
  TYPE_FIELDS (t) = fieldlist;

  layout_type (t);
}

/* Construct a playback::type instance (wrapping a tree) for a function
   type.  */

playback::type *
playback::context::
new_function_type (type *return_type,
		   const auto_vec<type *> *param_types,
		   int is_variadic)
{
  int i;
  type *param_type;

  tree *arg_types = (tree *)xcalloc(param_types->length (), sizeof(tree*));

  FOR_EACH_VEC_ELT (*param_types, i, param_type)
    arg_types[i] = param_type->as_tree ();

  tree fn_type;
  if (is_variadic)
    fn_type =
      build_varargs_function_type_array (return_type->as_tree (),
					 param_types->length (),
					 arg_types);
  else
    fn_type = build_function_type_array (return_type->as_tree (),
					 param_types->length (),
					 arg_types);
  free (arg_types);

  return new type (fn_type);
}

/* Construct a playback::param instance (wrapping a tree).  */

playback::param *
playback::context::
new_param (location *loc,
	   type *type,
	   const char *name)
{
  gcc_assert (type);
  gcc_assert (name);
  tree inner = build_decl (UNKNOWN_LOCATION, PARM_DECL,
			   get_identifier (name), type->as_tree ());
  if (loc)
    set_tree_location (inner, loc);

  return new param (this, inner);
}

const char* fn_attribute_to_string (gcc_jit_fn_attribute attr)
{
  switch (attr)
  {
    case GCC_JIT_FN_ATTRIBUTE_ALIAS:
      return "alias";
    case GCC_JIT_FN_ATTRIBUTE_ALWAYS_INLINE:
      return "always_inline";
    case GCC_JIT_FN_ATTRIBUTE_INLINE:
      return NULL;
    case GCC_JIT_FN_ATTRIBUTE_NOINLINE:
      return "noinline";
    case GCC_JIT_FN_ATTRIBUTE_TARGET:
      return "target";
    case GCC_JIT_FN_ATTRIBUTE_USED:
      return "used";
    case GCC_JIT_FN_ATTRIBUTE_VISIBILITY:
      return "visibility";
    case GCC_JIT_FN_ATTRIBUTE_COLD:
      return "cold";
    case GCC_JIT_FN_ATTRIBUTE_RETURNS_TWICE:
      return "returns_twice";
    case GCC_JIT_FN_ATTRIBUTE_PURE:
      return "pure";
    case GCC_JIT_FN_ATTRIBUTE_CONST:
      return "const";
    case GCC_JIT_FN_ATTRIBUTE_WEAK:
      return "weak";
    case GCC_JIT_FN_ATTRIBUTE_NONNULL:
      return "nonnull";
    case GCC_JIT_FN_ATTRIBUTE_ARM_CMSE_NONSECURE_CALL:
      return "cmse_nonsecure_call";
    case GCC_JIT_FN_ATTRIBUTE_ARM_CMSE_NONSECURE_ENTRY:
      return "cmse_nonsecure_entry";
    case GCC_JIT_FN_ATTRIBUTE_ARM_PCS:
      return "pcs";
    case GCC_JIT_FN_ATTRIBUTE_AVR_INTERRUPT:
      return "interrupt";
    case GCC_JIT_FN_ATTRIBUTE_AVR_NOBLOCK:
      return "noblock";
    case GCC_JIT_FN_ATTRIBUTE_AVR_SIGNAL:
      return "signal";
    case GCC_JIT_FN_ATTRIBUTE_GCN_AMDGPU_HSA_KERNEL:
      return "amdgpu_hsa_kernel";
    case GCC_JIT_FN_ATTRIBUTE_MSP430_INTERRUPT:
      return "interrupt";
    case GCC_JIT_FN_ATTRIBUTE_NVPTX_KERNEL:
      return "kernel";
    case GCC_JIT_FN_ATTRIBUTE_RISCV_INTERRUPT:
      return "interrupt";
    case GCC_JIT_FN_ATTRIBUTE_X86_FAST_CALL:
      return "fastcall";
    case GCC_JIT_FN_ATTRIBUTE_X86_INTERRUPT:
      return "interrupt";
    case GCC_JIT_FN_ATTRIBUTE_X86_MS_ABI:
      return "ms_abi";
    case GCC_JIT_FN_ATTRIBUTE_X86_STDCALL:
      return "stdcall";
    case GCC_JIT_FN_ATTRIBUTE_X86_SYSV_ABI:
      return "sysv_abi";
    case GCC_JIT_FN_ATTRIBUTE_X86_THIS_CALL:
      return "thiscall";
    case GCC_JIT_FN_ATTRIBUTE_MAX:
      return NULL;
  }
  return NULL;
}

const char* variable_attribute_to_string (gcc_jit_variable_attribute attr)
{
  switch (attr)
  {
    case GCC_JIT_VARIABLE_ATTRIBUTE_VISIBILITY:
      return "visibility";
    case GCC_JIT_VARIABLE_ATTRIBUTE_MAX:
      return NULL;
  }
  return NULL;
}

/* Construct a playback::function instance.  */

playback::function *
playback::context::
new_function (location *loc,
	      enum gcc_jit_function_kind kind,
	      type *return_type,
	      const char *name,
	      const auto_vec<param *> *params,
	      int is_variadic,
	      enum built_in_function builtin_id,
	      const std::vector<gcc_jit_fn_attribute> &attributes,
	      const std::vector<std::pair<gcc_jit_fn_attribute,
					  std::string>> &string_attributes,
	      const std::vector<std::pair<gcc_jit_fn_attribute,
					  std::vector<int>>>
					  &int_array_attributes,
	      bool is_target_builtin)
{
  int i;
  param *param;

  //can return_type be NULL?
  gcc_assert (name);

  tree *arg_types = (tree *)xcalloc(params->length (), sizeof(tree*));
  FOR_EACH_VEC_ELT (*params, i, param)
    arg_types[i] = TREE_TYPE (param->as_tree ());

  tree fn_type;
  if (is_variadic)
    fn_type = build_varargs_function_type_array (return_type->as_tree (),
						 params->length (), arg_types);
  else
    fn_type = build_function_type_array (return_type->as_tree (),
					 params->length (), arg_types);
  free (arg_types);

  /* FIXME: this uses input_location: */
  tree fndecl = build_fn_decl (name, fn_type);

  if (loc)
    set_tree_location (fndecl, loc);

  tree resdecl = build_decl (UNKNOWN_LOCATION, RESULT_DECL,
			     NULL_TREE, return_type->as_tree ());
  DECL_ARTIFICIAL (resdecl) = 1;
  DECL_IGNORED_P (resdecl) = 1;
  DECL_RESULT (fndecl) = resdecl;
  DECL_CONTEXT (resdecl) = fndecl;

  tree fn_attributes = NULL_TREE;

  if (is_target_builtin)
  {
    tree *decl = target_builtins.get (name);
    if (decl != NULL)
      fndecl = *decl;
    else
      add_error (loc, "cannot find target builtin %s", name);
  }

  if (builtin_id)
    {
      gcc_assert (loc == NULL);
      DECL_SOURCE_LOCATION (fndecl) = BUILTINS_LOCATION;

      built_in_class fclass = builtins_manager::get_class (builtin_id);
      set_decl_built_in_function (fndecl, fclass, builtin_id);
      set_builtin_decl (builtin_id, fndecl,
			builtins_manager::implicit_p (builtin_id));

      builtins_manager *bm = get_builtins_manager ();
      tree attrs = bm->get_attrs_tree (builtin_id);
      if (attrs)
	decl_attributes (&fndecl, attrs, ATTR_FLAG_BUILT_IN);
      else
	decl_attributes (&fndecl, NULL_TREE, 0);
    }

  if (kind != GCC_JIT_FUNCTION_IMPORTED)
    {
      tree param_decl_list = NULL;
      FOR_EACH_VEC_ELT (*params, i, param)
	{
	  param_decl_list = chainon (param->as_tree (), param_decl_list);
	}

      /* The param list was created in reverse order; fix it: */
      param_decl_list = nreverse (param_decl_list);

      tree t;
      for (t = param_decl_list; t; t = DECL_CHAIN (t))
	{
	  DECL_CONTEXT (t) = fndecl;
	  DECL_ARG_TYPE (t) = TREE_TYPE (t);
	}

      /* Set it up on DECL_ARGUMENTS */
      DECL_ARGUMENTS(fndecl) = param_decl_list;
    }

  if (kind == GCC_JIT_FUNCTION_ALWAYS_INLINE)
    {
      DECL_DECLARED_INLINE_P (fndecl) = 1;

      /* Add attribute "always_inline": */
      fn_attributes = tree_cons (get_identifier ("always_inline"),
				 NULL,
				 fn_attributes);
    }

  /* All attributes need to be declared in `dummy-frontend.cc` and more
     specifically in `jit_attribute_table`. */
  for (auto attr: attributes)
  {
    if (attr == GCC_JIT_FN_ATTRIBUTE_INLINE)
      DECL_DECLARED_INLINE_P (fndecl) = 1;

    const char* attribute = fn_attribute_to_string (attr);
    if (attribute)
    {
      tree ident = get_identifier (attribute);
      fn_attributes = tree_cons (ident, NULL_TREE, fn_attributes);
    }
  }

  for (auto attr: string_attributes)
  {
    gcc_jit_fn_attribute& name = std::get<0>(attr);
    std::string& value = std::get<1>(attr);
    tree attribute_value = build_tree_list (NULL_TREE,
	::build_string (value.length () + 1, value.c_str ()));
    const char* attribute = fn_attribute_to_string (name);
    tree ident = attribute ? get_identifier (attribute) : NULL;

    if (ident)
      fn_attributes = tree_cons (ident, attribute_value, fn_attributes);
  }

  for (auto attr: int_array_attributes)
  {
    gcc_jit_fn_attribute& name = std::get<0>(attr);
    std::vector<int>& values = std::get<1>(attr);

    const char* attribute = fn_attribute_to_string (name);
    tree ident = attribute ? get_identifier (attribute) : NULL;

    if (!ident)
      continue;

    tree tree_list = NULL_TREE;
    tree *p_tree_list = &tree_list;
    for (auto value : values)
    {
      tree int_value = build_int_cst (integer_type_node, value);
      *p_tree_list = build_tree_list (NULL, int_value);
      p_tree_list = &TREE_CHAIN (*p_tree_list);
    }
    fn_attributes = tree_cons (ident, tree_list, fn_attributes);
  }

  decl_attributes (&fndecl, fn_attributes, 0);
  function *func = new function (this, fndecl, kind);
  m_functions.safe_push (func);
  return func;
}

/* In use by new_global and new_global_initialized.  */

tree
playback::context::
global_new_decl (location *loc,
		 enum gcc_jit_global_kind kind,
		 type *type,
		 const char *name,
		 enum global_var_flags flags,
		 const std::vector<std::pair<gcc_jit_variable_attribute,
					     std::string>> &attributes,
		 bool readonly)
{
  gcc_assert (type);
  gcc_assert (name);

  tree type_tree = type->as_tree ();

  tree inner = build_decl (UNKNOWN_LOCATION, VAR_DECL,
			   get_identifier (name),
			   type_tree);

  TREE_PUBLIC (inner) = (kind != GCC_JIT_GLOBAL_INTERNAL);


  int will_be_init = flags & (GLOBAL_VAR_FLAGS_WILL_BE_RVAL_INIT |
			      GLOBAL_VAR_FLAGS_WILL_BE_BLOB_INIT);

  /* A VAR_DECL with DECL_INITIAL will not end up in .common section.  */
  if (!will_be_init)
    DECL_COMMON (inner) = 1;

  switch (kind)
    {
    default:
      gcc_unreachable ();

    case GCC_JIT_GLOBAL_EXPORTED:
      TREE_STATIC (inner) = 1;
      break;

    case GCC_JIT_GLOBAL_INTERNAL:
      TREE_STATIC (inner) = 1;
      break;

    case GCC_JIT_GLOBAL_IMPORTED:
      DECL_EXTERNAL (inner) = 1;
      break;
    }

  if (TYPE_READONLY (type_tree) || readonly)
    TREE_READONLY (inner) = 1;

  if (loc)
    set_tree_location (inner, loc);

  set_variable_string_attribute (attributes, inner);

  return inner;
}

void
playback::
set_variable_string_attribute (
  const std::vector<std::pair<gcc_jit_variable_attribute,
			       std::string>> &string_attributes,
  tree decl)
{
  tree var_attributes = NULL_TREE;
  for (auto attr: string_attributes)
  {
    gcc_jit_variable_attribute& name = std::get<0>(attr);
    std::string& value = std::get<1>(attr);
    tree attribute_value = build_tree_list (NULL_TREE,
	::build_string (value.length () + 1, value.c_str ()));
    tree ident = get_identifier (variable_attribute_to_string (name));
    if (ident)
      var_attributes = tree_cons (ident, attribute_value, var_attributes);
  }
  decl_attributes (&decl, var_attributes, 0);
}

/* In use by new_global and new_global_initialized.  */

playback::lvalue *
playback::context::
global_finalize_lvalue (tree inner)
{
  m_globals.safe_push (inner);

  return new lvalue (this, inner);
}

/* Construct a playback::lvalue instance (wrapping a tree).  */

playback::lvalue *
playback::context::
new_global (location *loc,
	    enum gcc_jit_global_kind kind,
	    type *type,
	    const char *name,
	    enum global_var_flags flags,
	    const std::vector<std::pair<gcc_jit_variable_attribute,
					std::string>> &attributes,
	    bool readonly)
{
  tree inner =
    global_new_decl (loc, kind, type, name, flags, attributes, readonly);

  return global_finalize_lvalue (inner);
}

void
playback::context::
global_set_init_rvalue (lvalue* variable,
			rvalue* init)
{
  tree inner = variable->as_tree ();

  /* We need to fold all expressions as much as possible.  The code
     for a DECL_INITIAL only handles some operations,
     etc addition, minus, 'address of'.  See output_addressed_constants ()
     in varasm.cc.  */
  tree init_tree = init->as_tree ();
  tree folded = fold_const_var (init_tree);

  if (!TREE_CONSTANT (folded))
    {
      tree name = DECL_NAME (inner);

      if (name != NULL_TREE)
	add_error (NULL,
		   "unable to convert initial value for the global variable %s"
		   " to a compile-time constant",
		   IDENTIFIER_POINTER (name));
      else
	add_error (NULL,
		   "unable to convert initial value for global variable"
		   " to a compile-time constant");
      return;
    }

  DECL_INITIAL (inner) = folded;
}

playback::rvalue *
playback::context::
new_ctor (location *loc,
	  type *type,
	  const auto_vec<field*> *fields,
	  const auto_vec<rvalue*> *rvalues)
{
  tree type_tree = type->as_tree ();

  /* Handle empty ctors first.  I.e. set everything to 0.  */
  if (rvalues->length () == 0)
    return new rvalue (this, build_constructor (type_tree, NULL));

  /* Handle arrays (and return).  */
  if (TREE_CODE (type_tree) == ARRAY_TYPE)
    {
      int n = rvalues->length ();
      /* The vec for the constructor node.  */
      vec<constructor_elt, va_gc> *v = NULL;
      vec_alloc (v, n);

      for (int i = 0; i < n; i++)
	{
	  rvalue *rv = (*rvalues)[i];
	  /* null rvalues indicate that the element should be zeroed.  */
	  if (rv)
	    CONSTRUCTOR_APPEND_ELT (v,
				    build_int_cst (size_type_node, i),
				    rv->as_tree ());
	  else
	    CONSTRUCTOR_APPEND_ELT (v,
				    build_int_cst (size_type_node, i),
				    build_zero_cst (TREE_TYPE (type_tree)));
	}

      tree ctor = build_constructor (type_tree, v);

      if (loc)
	set_tree_location (ctor, loc);

      return new rvalue (this, ctor);
    }

  /* Handle structs and unions.  */
  int n = fields->length ();

  /* The vec for the constructor node.  */
  vec<constructor_elt, va_gc> *v = NULL;
  vec_alloc (v, n);

  /* Iterate over the fields, building initializations.  */
  for (int i = 0;i < n; i++)
    {
      tree field = (*fields)[i]->as_tree ();
      rvalue *rv = (*rvalues)[i];
      /* If the value is NULL, it means we should zero the field.  */
      if (rv)
	CONSTRUCTOR_APPEND_ELT (v, field, rv->as_tree ());
      else
	{
	  tree zero_cst = build_zero_cst (TREE_TYPE (field));
	  CONSTRUCTOR_APPEND_ELT (v, field, zero_cst);
	}
    }

  tree ctor = build_constructor (type_tree, v);

  if (loc)
    set_tree_location (ctor, loc);

  return new rvalue (this, build_constructor (type_tree, v));
}

/* Fill 'constructor_elements' with the memory content of
   'initializer'.  Each element of the initializer is of the size of
   type T.  In use by new_global_initialized.*/

template<typename T>
static void
load_blob_in_ctor (vec<constructor_elt, va_gc> *&constructor_elements,
		   size_t num_elem,
		   const void *initializer)
{
  /* Loosely based on 'output_init_element' c-typeck.cc:9691.  */
  const T *p = (const T *)initializer;
  tree node = make_unsigned_type (BITS_PER_UNIT * sizeof (T));
  for (size_t i = 0; i < num_elem; i++)
    {
      constructor_elt celt =
	{ build_int_cst (long_unsigned_type_node, i),
	  build_int_cst (node, p[i]) };
      vec_safe_push (constructor_elements, celt);
    }
}

/* Construct an initialized playback::lvalue instance (wrapping a
   tree).  */

playback::lvalue *
playback::context::
new_global_initialized (location *loc,
			enum gcc_jit_global_kind kind,
			type *type,
			size_t element_size,
			size_t initializer_num_elem,
			const void *initializer,
			const char *name,
			enum global_var_flags flags,
			const std::vector<std::pair<gcc_jit_variable_attribute,
						    std::string>> &attributes,
			bool readonly)
{
  tree inner = global_new_decl (loc, kind, type, name, flags, attributes, readonly);

  vec<constructor_elt, va_gc> *constructor_elements = NULL;

  switch (element_size)
    {
    case 1:
      load_blob_in_ctor<uint8_t> (constructor_elements, initializer_num_elem,
				  initializer);
      break;
    case 2:
      load_blob_in_ctor<uint16_t> (constructor_elements, initializer_num_elem,
				   initializer);
      break;
    case 4:
      load_blob_in_ctor<uint32_t> (constructor_elements, initializer_num_elem,
				   initializer);
      break;
    case 8:
      load_blob_in_ctor<uint64_t> (constructor_elements, initializer_num_elem,
				   initializer);
      break;
    default:
      /* This function is serving on sizes returned by 'get_size',
	 these are all covered by the previous cases.  */
      gcc_unreachable ();
    }
  /* Compare with 'pop_init_level' c-typeck.cc:8780.  */
  tree ctor = build_constructor (type->as_tree (), constructor_elements);
  constructor_elements = NULL;

  /* Compare with 'store_init_value' c-typeck.cc:7555.  */
  DECL_INITIAL (inner) = ctor;

  return global_finalize_lvalue (inner);
}

/* Implementation of the various
      gcc::jit::playback::context::new_rvalue_from_const <HOST_TYPE>
   methods.
   Each of these constructs a playback::rvalue instance (wrapping a tree).

   These specializations are required to be in the same namespace
   as the template, hence we now have to enter the gcc::jit::playback
   namespace.  */

namespace playback
{

/* Specialization of making an rvalue from a const, for host <int>.  */

template <>
rvalue *
context::
new_rvalue_from_const <int> (type *type,
			     int value)
{
  // FIXME: type-checking, or coercion?
  tree inner_type = type->as_tree ();
  if (INTEGRAL_TYPE_P (inner_type))
    {
      tree inner = build_int_cst (inner_type, value);
      return new rvalue (this, inner);
    }
  else
    {
      REAL_VALUE_TYPE real_value;
      real_from_integer (&real_value, VOIDmode, value, SIGNED);
      tree inner = build_real (inner_type, real_value);
      return new rvalue (this, inner);
    }
}

/* Specialization of making an rvalue from a const, for host <long>.  */

template <>
rvalue *
context::
new_rvalue_from_const <long> (type *type,
			      long value)
{
  // FIXME: type-checking, or coercion?
  tree inner_type = type->as_tree ();
  if (INTEGRAL_TYPE_P (inner_type))
    {
      tree inner = build_int_cst (inner_type, value);
      return new rvalue (this, inner);
    }
  else
    {
      REAL_VALUE_TYPE real_value;
      real_from_integer (&real_value, VOIDmode, value, SIGNED);
      tree inner = build_real (inner_type, real_value);
      return new rvalue (this, inner);
    }
}

/* Specialization of making an rvalue from a const, for host <double>.  */

template <>
rvalue *
context::
new_rvalue_from_const <double> (type *type,
				double value)
{
  // FIXME: type-checking, or coercion?
  tree inner_type = type->as_tree ();

  mpfr_t mpf_value;

  mpfr_init2 (mpf_value, 64);
  mpfr_set_d (mpf_value, value, MPFR_RNDN);

  /* We have a "double", we want a REAL_VALUE_TYPE.

     realmpfr.cc:real_from_mpfr.  */
  REAL_VALUE_TYPE real_value;
  real_from_mpfr (&real_value, mpf_value, inner_type, MPFR_RNDN);
  tree inner = build_real (inner_type, real_value);
  return new rvalue (this, inner);
}

/* Specialization of making an rvalue from a const, for host <void *>.  */

template <>
rvalue *
context::
new_rvalue_from_const <void *> (type *type,
				void *value)
{
  tree inner_type = type->as_tree ();
  /* FIXME: how to ensure we have a wide enough type?  */
  tree inner = build_int_cstu (inner_type, (unsigned HOST_WIDE_INT)value);
  return new rvalue (this, inner);
}

/* We're done implementing the specializations of
      gcc::jit::playback::context::new_rvalue_from_const <T>
   so we can exit the gcc::jit::playback namespace.  */

} // namespace playback

/* Construct a playback::rvalue instance (wrapping a tree).  */

playback::rvalue *
playback::context::
new_sizeof (type *type)
{
  tree inner = TYPE_SIZE_UNIT (type->as_tree ());
  return new rvalue (this, inner);
}

/* Construct a playback::rvalue instance (wrapping a tree).  */

playback::rvalue *
playback::context::
new_alignof (type *type)
{
  int alignment = TYPE_ALIGN (type->as_tree ()) / BITS_PER_UNIT;
  tree inner = build_int_cst (integer_type_node, alignment);
  return new rvalue (this, inner);
}

/* Construct a playback::rvalue instance (wrapping a tree).  */

playback::rvalue *
playback::context::
new_string_literal (const char *value)
{
  /* Compare with c-family/c-common.cc: fix_string_type.  */
  size_t len = strlen (value);
  tree i_type = build_index_type (size_int (len));
  tree a_type = build_array_type (char_type_node, i_type);
  /* build_string len parameter must include NUL terminator when
     building C strings.  */
  tree t_str = ::build_string (len + 1, value);
  TREE_TYPE (t_str) = a_type;

  /* Convert to (const char*), loosely based on
     c/c-typeck.cc: array_to_pointer_conversion,
     by taking address of start of string.  */
  tree t_addr = build1 (ADDR_EXPR, m_const_char_ptr, t_str);

  return new rvalue (this, t_addr);
}

/* Construct a playback::rvalue instance (wrapping a tree) for a
   vector.  */

playback::rvalue *
playback::context::new_rvalue_from_vector (location *,
					   type *type,
					   const auto_vec<rvalue *> &elements)
{
  vec<constructor_elt, va_gc> *v;
  vec_alloc (v, elements.length ());
  for (unsigned i = 0; i < elements.length (); ++i)
    CONSTRUCTOR_APPEND_ELT (v, NULL_TREE, elements[i]->as_tree ());
  tree t_ctor = build_constructor (type->as_tree (), v);
  return new rvalue (this, t_ctor);
}

/* Construct a playback::rvalue instance (wrapping a tree) for a
   vector perm.  */

playback::rvalue *
playback::context::new_rvalue_vector_perm (location *loc,
					   rvalue* elements1,
					   rvalue* elements2,
					   rvalue* mask)
{
  tree t_elements1 = elements1->as_tree ();
  tree t_elements2 = elements2->as_tree ();
  tree t_mask = mask->as_tree ();

  tree t_vector_perm = build3 (VEC_PERM_EXPR, TREE_TYPE (t_elements1),
			       t_elements1, t_elements2, t_mask);
  if (loc)
    set_tree_location (t_vector_perm, loc);
  return new rvalue (this, t_vector_perm);
}

/* Coerce a tree expression into a boolean tree expression.  */

tree
playback::context::
as_truth_value (tree expr, location *loc)
{
  /* Compare to c-typeck.cc:c_objc_common_truthvalue_conversion */
  tree typed_zero = fold_build1 (CONVERT_EXPR,
				 TREE_TYPE (expr),
				 integer_zero_node);
  if (loc)
    set_tree_location (typed_zero, loc);

  tree type = TREE_TYPE (expr);
  expr = fold_build2_loc (UNKNOWN_LOCATION,
    NE_EXPR, type, expr, typed_zero);
  if (loc)
    set_tree_location (expr, loc);

  return expr;
}

/* Add a "top-level" basic asm statement (i.e. one outside of any functions)
   containing ASM_STMTS.

   Compare with c_parser_asm_definition.  */

void
playback::context::add_top_level_asm (const char *asm_stmts)
{
  tree asm_str = build_string (asm_stmts);
  symtab->finalize_toplevel_asm (asm_str);
}

/* Construct a playback::rvalue instance (wrapping a tree) for a
   unary op.  */

playback::rvalue *
playback::context::
new_unary_op (location *loc,
	      enum gcc_jit_unary_op op,
	      type *result_type,
	      rvalue *a)
{
  // FIXME: type-checking, or coercion?
  enum tree_code inner_op;

  gcc_assert (result_type);
  gcc_assert (a);

  tree node = a->as_tree ();
  node = fold_const_var (node);

  tree inner_result = NULL;

  switch (op)
    {
    default:
      add_error (loc, "unrecognized (enum gcc_jit_unary_op) value: %i", op);
      return NULL;

    case GCC_JIT_UNARY_OP_MINUS:
      inner_op = NEGATE_EXPR;
      break;

    case GCC_JIT_UNARY_OP_BITWISE_NEGATE:
      inner_op = BIT_NOT_EXPR;
      break;

    case GCC_JIT_UNARY_OP_LOGICAL_NEGATE:
      node = as_truth_value (node, loc);
      inner_result = invert_truthvalue (node);
      if (loc)
	set_tree_location (inner_result, loc);
      return new rvalue (this, inner_result);

    case GCC_JIT_UNARY_OP_ABS:
      inner_op = ABS_EXPR;
      break;
    }

  inner_result = build1 (inner_op,
			 result_type->as_tree (),
			 node);

  /* Try to fold.  */
  inner_result = fold (inner_result);

  if (loc)
    set_tree_location (inner_result, loc);

  return new rvalue (this, inner_result);
}

/* Construct a playback::rvalue instance (wrapping a tree) for a
   binary op.  */

playback::rvalue *
playback::context::
new_binary_op (location *loc,
	       enum gcc_jit_binary_op op,
	       type *result_type,
	       rvalue *a, rvalue *b)
{
  // FIXME: type-checking, or coercion?
  enum tree_code inner_op;

  gcc_assert (result_type);
  gcc_assert (a);
  gcc_assert (b);

  tree node_a = a->as_tree ();
  node_a = fold_const_var (node_a);

  tree node_b = b->as_tree ();
  node_b = fold_const_var (node_b);

  switch (op)
    {
    default:
      add_error (loc, "unrecognized (enum gcc_jit_binary_op) value: %i", op);
      return NULL;

    case GCC_JIT_BINARY_OP_PLUS:
      inner_op = PLUS_EXPR;
      break;

    case GCC_JIT_BINARY_OP_MINUS:
      inner_op = MINUS_EXPR;
      break;

    case GCC_JIT_BINARY_OP_MULT:
      inner_op = MULT_EXPR;
      break;

    case GCC_JIT_BINARY_OP_DIVIDE:
      if (FLOAT_TYPE_P (result_type->as_tree ()))
	/* Floating-point division: */
	inner_op = RDIV_EXPR;
      else
	/* Truncating to zero: */
	inner_op = TRUNC_DIV_EXPR;
      break;

    case GCC_JIT_BINARY_OP_MODULO:
      inner_op = TRUNC_MOD_EXPR;
      break;

    case GCC_JIT_BINARY_OP_BITWISE_AND:
      inner_op = BIT_AND_EXPR;
      break;

    case GCC_JIT_BINARY_OP_BITWISE_XOR:
      inner_op = BIT_XOR_EXPR;
      break;

    case GCC_JIT_BINARY_OP_BITWISE_OR:
      inner_op = BIT_IOR_EXPR;
      break;

    case GCC_JIT_BINARY_OP_LOGICAL_AND:
      node_a = as_truth_value (node_a, loc);
      node_b = as_truth_value (node_b, loc);
      inner_op = TRUTH_ANDIF_EXPR;
      break;

    case GCC_JIT_BINARY_OP_LOGICAL_OR:
      node_a = as_truth_value (node_a, loc);
      node_b = as_truth_value (node_b, loc);
      inner_op = TRUTH_ORIF_EXPR;
      break;

    case GCC_JIT_BINARY_OP_LSHIFT:
      inner_op = LSHIFT_EXPR;
      break;

    case GCC_JIT_BINARY_OP_RSHIFT:
      inner_op = RSHIFT_EXPR;
      break;
    }

  tree inner_expr = build2 (inner_op,
			    result_type->as_tree (),
			    node_a,
			    node_b);

  /* Try to fold the expression.  */
  inner_expr = fold (inner_expr);

  if (loc)
    set_tree_location (inner_expr, loc);

  return new rvalue (this, inner_expr);
}

/* Construct a playback::rvalue instance (wrapping a tree) for a
   comparison.  */

playback::rvalue *
playback::context::
new_comparison (location *loc,
		enum gcc_jit_comparison op,
		rvalue *a, rvalue *b, type *vec_result_type)
{
  // FIXME: type-checking, or coercion?
  enum tree_code inner_op;

  gcc_assert (a);
  gcc_assert (b);

  switch (op)
    {
    default:
      add_error (loc, "unrecognized (enum gcc_jit_comparison) value: %i", op);
      return NULL;

    case GCC_JIT_COMPARISON_EQ:
      inner_op = EQ_EXPR;
      break;
    case GCC_JIT_COMPARISON_NE:
      inner_op = NE_EXPR;
      break;
    case GCC_JIT_COMPARISON_LT:
      inner_op = LT_EXPR;
      break;
    case GCC_JIT_COMPARISON_LE:
      inner_op = LE_EXPR;
      break;
    case GCC_JIT_COMPARISON_GT:
      inner_op = GT_EXPR;
      break;
    case GCC_JIT_COMPARISON_GE:
      inner_op = GE_EXPR;
      break;
    }

  tree node_a = a->as_tree ();
  node_a = fold_const_var (node_a);
  tree node_b = b->as_tree ();
  node_b = fold_const_var (node_b);

  tree inner_expr;
  tree a_type = TREE_TYPE (node_a);
  if (VECTOR_TYPE_P (a_type))
  {
    /* Build a vector comparison.  See build_vec_cmp in c-typeck.cc for
       reference.  */
    tree t_vec_result_type = vec_result_type->as_tree ();
    tree zero_vec = build_zero_cst (t_vec_result_type);
    tree minus_one_vec = build_minus_one_cst (t_vec_result_type);
    tree cmp_type = truth_type_for (a_type);
    tree cmp = build2 (inner_op, cmp_type, node_a, node_b);
    inner_expr = build3 (VEC_COND_EXPR, t_vec_result_type, cmp, minus_one_vec,
			 zero_vec);
  }
  else
  {
    inner_expr = build2 (inner_op,
			 boolean_type_node,
			 node_a,
			 node_b);
  }

  /* Try to fold.  */
  inner_expr = fold (inner_expr);

  if (loc)
    set_tree_location (inner_expr, loc);
  return new rvalue (this, inner_expr);
}

/* Construct a playback::rvalue instance (wrapping a tree) for a
   function call.  */

playback::rvalue *
playback::context::
build_call (location *loc,
	    tree fn_ptr,
	    const auto_vec<rvalue *> *args,
	    bool require_tail_call)
{
  vec<tree, va_gc> *tree_args;
  vec_alloc (tree_args, args->length ());
  for (unsigned i = 0; i < args->length (); i++)
    tree_args->quick_push ((*args)[i]->as_tree ());

  if (loc)
    set_tree_location (fn_ptr, loc);

  tree fn = TREE_TYPE (fn_ptr);
  tree fn_type = TREE_TYPE (fn);
  tree return_type = TREE_TYPE (fn_type);

  tree call = build_call_vec (return_type,
			      fn_ptr, tree_args);

  if (require_tail_call)
    CALL_EXPR_MUST_TAIL_CALL (call) = 1;

  return new rvalue (this, call);

  /* see c-typeck.cc: build_function_call
     which calls build_function_call_vec

     which does lots of checking, then:
    result = build_call_array_loc (loc, TREE_TYPE (fntype),
				   function, nargs, argarray);
    which is in tree.cc
    (see also build_call_vec)
   */
}

/* Construct a playback::rvalue instance (wrapping a tree) for a
   call to a specific function.  */

playback::rvalue *
playback::context::
new_call (location *loc,
	  function *func,
	  const auto_vec<rvalue *> *args,
	  bool require_tail_call)
{
  tree fndecl;

  gcc_assert (func);

  fndecl = func->as_fndecl ();

  tree fntype = TREE_TYPE (fndecl);

  tree fn = build1 (ADDR_EXPR, build_pointer_type (fntype), fndecl);

  return build_call (loc, fn, args, require_tail_call);
}

/* Construct a playback::rvalue instance (wrapping a tree) for a
   call through a function pointer.  */

playback::rvalue *
playback::context::
new_call_through_ptr (location *loc,
		      rvalue *fn_ptr,
		      const auto_vec<rvalue *> *args,
		      bool require_tail_call)
{
  gcc_assert (fn_ptr);
  tree t_fn_ptr = fn_ptr->as_tree ();

  return build_call (loc, t_fn_ptr, args, require_tail_call);
}

/* Construct a tree for a cast.  */

tree
playback::context::build_cast (playback::location *loc,
			       playback::rvalue *expr,
			       playback::type *type_)
{
  /* For comparison, see:
     - c/c-typeck.cc:build_c_cast
     - c/c-convert.cc: convert
     - convert.h

     Only some kinds of cast are currently supported here.  */
  tree t_expr = expr->as_tree ();
  t_expr = fold_const_var (t_expr);

  tree t_dst_type = type_->as_tree ();
  tree t_ret = NULL;
  t_ret = targetm.convert_to_type (t_dst_type, t_expr);
  if (t_ret)
      return t_ret;
  enum tree_code dst_code = TREE_CODE (t_dst_type);
  switch (dst_code)
    {
    case INTEGER_TYPE:
    case ENUMERAL_TYPE:
      t_ret = convert_to_integer (t_dst_type, t_expr);
      goto maybe_fold;

    case BOOLEAN_TYPE:
      /* Compare with c_objc_common_truthvalue_conversion and
	 c_common_truthvalue_conversion. */
      /* For now, convert to: (t_expr != 0)  */
      t_ret = build2 (NE_EXPR, t_dst_type,
		      t_expr,
		      build_int_cst (TREE_TYPE (t_expr), 0));
      goto maybe_fold;

    case REAL_TYPE:
      t_ret = convert_to_real (t_dst_type, t_expr);
      goto maybe_fold;

    case POINTER_TYPE:
      t_ret = build1 (NOP_EXPR, t_dst_type, t_expr);
      goto maybe_fold;

    default:
      add_error (loc, "couldn't handle cast during playback");
      fprintf (stderr, "input expression:\n");
      debug_tree (t_expr);
      fprintf (stderr, "requested type:\n");
      debug_tree (t_dst_type);
      return error_mark_node;

    maybe_fold:
      if (TREE_CODE (t_ret) != C_MAYBE_CONST_EXPR)
	t_ret = fold (t_ret);
      return t_ret;
    }
}

/* Construct a playback::rvalue instance (wrapping a tree) for a
   cast.  */

playback::rvalue *
playback::context::
new_cast (playback::location *loc,
	  playback::rvalue *expr,
	  playback::type *type_)
{

  tree t_cast = build_cast (loc, expr, type_);
  if (loc)
    set_tree_location (t_cast, loc);
  return new rvalue (this, t_cast);
}

/* Construct a playback::rvalue instance (wrapping a tree) for a
   bitcast.  */

playback::rvalue *
playback::context::
new_bitcast (location *loc,
	     rvalue *expr,
	     type *type_)
{
  tree expr_size = TYPE_SIZE (expr->get_type ()->as_tree ());
  tree type_size = TYPE_SIZE (type_->as_tree ());
  tree t_expr = expr->as_tree ();
  tree t_dst_type = type_->as_tree ();
  if (expr_size != type_size)
  {
    active_playback_ctxt->add_error (loc,
      "bitcast with types of different sizes");
    fprintf (stderr, "input expression (size: " HOST_WIDE_INT_PRINT_DEC "):\n",
	     tree_to_uhwi (expr_size));
    debug_tree (t_expr);
    fprintf (stderr, "requested type (size: " HOST_WIDE_INT_PRINT_DEC "):\n",
	     tree_to_uhwi (type_size));
    debug_tree (t_dst_type);
  }
  tree t_bitcast = build1 (VIEW_CONVERT_EXPR, t_dst_type, t_expr);
  if (loc)
    set_tree_location (t_bitcast, loc);
  return new rvalue (this, t_bitcast);
}

/* Construct a playback::lvalue instance (wrapping a tree) for an
   array access.  */

playback::lvalue *
playback::context::
new_array_access (location *loc,
		  rvalue *ptr,
		  rvalue *index)
{
  gcc_assert (ptr);
  gcc_assert (index);

  /* For comparison, see:
       c/c-typeck.cc: build_array_ref
       c-family/c-common.cc: pointer_int_sum
  */
  tree t_ptr = ptr->as_tree ();
  t_ptr = fold_const_var (t_ptr);
  tree t_index = index->as_tree ();
  t_index = fold_const_var (t_index);

  tree t_type_ptr = TREE_TYPE (t_ptr);
  tree t_type_star_ptr = TREE_TYPE (t_type_ptr);

  if (TREE_CODE (t_type_ptr) == ARRAY_TYPE)
    {
      tree t_result = build4 (ARRAY_REF, t_type_star_ptr, t_ptr, t_index,
			      NULL_TREE, NULL_TREE);
      t_result = fold (t_result);
      if (loc)
	set_tree_location (t_result, loc);
      return new lvalue (this, t_result);
    }
  else
    {
      /* Convert index to an offset in bytes.  */
      tree t_sizeof = size_in_bytes (t_type_star_ptr);
      t_index = fold_build1 (CONVERT_EXPR, sizetype, t_index);
      tree t_offset = fold_build2_loc (UNKNOWN_LOCATION,
	MULT_EXPR, sizetype, t_index, t_sizeof);

      /* Locate (ptr + offset).  */
      tree t_address = fold_build2_loc (UNKNOWN_LOCATION,
	POINTER_PLUS_EXPR, t_type_ptr, t_ptr, t_offset);

      tree t_indirection = fold_build1 (INDIRECT_REF, t_type_star_ptr, t_address);
      if (loc)
	{
	  set_tree_location (t_sizeof, loc);
	  set_tree_location (t_offset, loc);
	  set_tree_location (t_address, loc);
	  set_tree_location (t_indirection, loc);
	}

      return new lvalue (this, t_indirection);
    }
}

/* Construct a playback::rvalue instance (wrapping a tree) for a
   vector conversion.  */

playback::rvalue *
playback::context::
convert_vector (location *loc,
		   rvalue *vector,
		   type *type)
{
  gcc_assert (vector);
  gcc_assert (type);

  /* For comparison, see:
       c/c-common.cc: c_build_vec_convert
  */

  tree t_vector = vector->as_tree ();

  tree t_result =
    build_call_expr_internal_loc (UNKNOWN_LOCATION, IFN_VEC_CONVERT,
      type->as_tree (), 1, t_vector);

  if (loc)
    set_tree_location (t_result, loc);

  return new rvalue (this, t_result);
}

/* The following functions come from c-common.h.  */
/* Like c_mark_addressable but don't check register qualifier.  */
void
common_mark_addressable_vec (tree t)
{
  while (handled_component_p (t) || TREE_CODE (t) == C_MAYBE_CONST_EXPR)
    {
      t = TREE_OPERAND (t, 0);
    }
  if (!VAR_P (t)
      && TREE_CODE (t) != PARM_DECL
      && TREE_CODE (t) != COMPOUND_LITERAL_EXPR
      && TREE_CODE (t) != TARGET_EXPR)
    return;
  if (!VAR_P (t) || !DECL_HARD_REGISTER (t))
    TREE_ADDRESSABLE (t) = 1;
  if (TREE_CODE (t) == COMPOUND_LITERAL_EXPR)
    TREE_ADDRESSABLE (COMPOUND_LITERAL_EXPR_DECL (t)) = 1;
  else if (TREE_CODE (t) == TARGET_EXPR)
    TREE_ADDRESSABLE (TARGET_EXPR_SLOT (t)) = 1;
}

/* Return true if TYPE is a vector type that should be subject to the GNU
   vector extensions (as opposed to a vector type that is used only for
   the purposes of defining target-specific built-in functions).  */

inline bool
gnu_vector_type_p (const_tree type)
{
  return TREE_CODE (type) == VECTOR_TYPE && !TYPE_INDIVISIBLE_P (type);
}

/* Return nonzero if REF is an lvalue valid for this language.
   Lvalues can be assigned, unless their type has TYPE_READONLY.
   Lvalues can have their address taken, unless they have C_DECL_REGISTER.  */

bool
lvalue_p (const_tree ref)
{
  const enum tree_code code = TREE_CODE (ref);

  switch (code)
    {
    case REALPART_EXPR:
    case IMAGPART_EXPR:
    case COMPONENT_REF:
      return lvalue_p (TREE_OPERAND (ref, 0));

    case C_MAYBE_CONST_EXPR:
      return lvalue_p (TREE_OPERAND (ref, 1));

    case COMPOUND_LITERAL_EXPR:
    case STRING_CST:
      return true;

    case MEM_REF:
    case TARGET_MEM_REF:
      /* MEM_REFs can appear from -fgimple parsing or folding, so allow them
	 here as well.  */
    case INDIRECT_REF:
    case ARRAY_REF:
    case VAR_DECL:
    case PARM_DECL:
    case RESULT_DECL:
    case ERROR_MARK:
      return (TREE_CODE (TREE_TYPE (ref)) != FUNCTION_TYPE
	      && TREE_CODE (TREE_TYPE (ref)) != METHOD_TYPE);

    case BIND_EXPR:
      return TREE_CODE (TREE_TYPE (ref)) == ARRAY_TYPE;

    default:
      return false;
    }
}

bool
convert_vector_to_array_for_subscript (tree *vecp)
{
  bool ret = false;
  if (gnu_vector_type_p (TREE_TYPE (*vecp)))
    {
      tree type = TREE_TYPE (*vecp);

      ret = !lvalue_p (*vecp);

      /* We are building an ARRAY_REF so mark the vector as addressable
	to not run into the gimplifiers premature setting of DECL_GIMPLE_REG_P
	 for function parameters.  */
      /* NOTE: that was the missing piece for making vector access work with
	optimizations enabled.  */
      common_mark_addressable_vec (*vecp);

      *vecp = build1 (VIEW_CONVERT_EXPR,
		      build_array_type_nelts (TREE_TYPE (type),
					      TYPE_VECTOR_SUBPARTS (type)),
		      *vecp);
    }
  return ret;
}

/* Construct a playback::lvalue instance (wrapping a tree) for a
   vector access.  */

playback::lvalue *
playback::context::
new_vector_access (location *loc,
		   rvalue *vector,
		   rvalue *index)
{
  gcc_assert (vector);
  gcc_assert (index);

  /* For comparison, see:
       c/c-typeck.cc: build_array_ref
  */

  tree t_vector = vector->as_tree ();
  bool non_lvalue = convert_vector_to_array_for_subscript (&t_vector);
  tree type = TREE_TYPE (TREE_TYPE (t_vector));
  tree t_result = build4 (ARRAY_REF, type, t_vector, index->as_tree (),
			  NULL_TREE, NULL_TREE);
  if (non_lvalue)
    t_result = non_lvalue (t_result);

  if (loc)
    set_tree_location (t_result, loc);
  return new lvalue (this, t_result);
}

/* Construct a tree for a field access.  */

tree
playback::context::
new_field_access (location *loc,
		  tree datum,
		  field *field)
{
  gcc_assert (datum);
  gcc_assert (field);

  /* Compare with c/c-typeck.cc:lookup_field, build_indirect_ref, and
     build_component_ref. */
  tree type = TREE_TYPE (datum);
  gcc_assert (type);
  gcc_assert (TREE_CODE (type) != POINTER_TYPE);

 tree t_field = field->as_tree ();
 tree ref = build3 (COMPONENT_REF, TREE_TYPE (t_field), datum,
		     t_field, NULL_TREE);
  if (loc)
    set_tree_location (ref, loc);
  return ref;
}

/* Construct a tree for a dereference.  */

tree
playback::context::
new_dereference (tree ptr,
		 location *loc)
{
  gcc_assert (ptr);

  tree type = TREE_TYPE (TREE_TYPE(ptr));
  tree datum = fold_build1 (INDIRECT_REF, type, ptr);
  if (loc)
    set_tree_location (datum, loc);
  return datum;
}

/* Construct a playback::type instance (wrapping a tree)
   with the given alignment.  */

playback::type *
playback::type::
get_aligned (size_t alignment_in_bytes) const
{
  tree t_new_type = build_variant_type_copy (m_inner);

  SET_TYPE_ALIGN (t_new_type, alignment_in_bytes * BITS_PER_UNIT);
  TYPE_USER_ALIGN (t_new_type) = 1;

  return new type (t_new_type);
}

/* Construct a playback::type instance (wrapping a tree)
   for the given vector type.  */

playback::type *
playback::type::
get_vector (size_t num_units) const
{
  tree t_new_type = build_vector_type (m_inner, num_units);
  return new type (t_new_type);
}

/* Construct a playback::lvalue instance (wrapping a tree) for a
   field access.  */

playback::lvalue *
playback::lvalue::
access_field (location *loc,
	      field *field)
{
  tree datum = as_tree ();
  tree ref = get_context ()->new_field_access (loc, datum, field);
  if (!ref)
    return NULL;
  return new lvalue (get_context (), ref);
}

/* Construct 