// Copyright Earl Warren <contact@earl-warren.org>
// Copyright Loïc Dachary <loic@dachary.org>
// SPDX-License-Identifier: MIT

package forgejo

import (
	"context"
	"fmt"

	"code.forgejo.org/f3/gof3/v3/f3"
	"code.forgejo.org/f3/gof3/v3/id"
	f3_tree "code.forgejo.org/f3/gof3/v3/tree/f3"
	"code.forgejo.org/f3/gof3/v3/tree/generic"
	"code.forgejo.org/f3/gof3/v3/util"

	forgejo_sdk "code.forgejo.org/f3/gof3/v3/forges/forgejo/sdk"
)

type review struct {
	common

	forgejoReview *forgejo_sdk.PullReview
}

var _ f3_tree.ForgeDriverInterface = &review{}

func newReview() generic.NodeDriverInterface {
	return &review{}
}

func (o *review) SetNative(review any) {
	o.forgejoReview = review.(*forgejo_sdk.PullReview)
}

func (o *review) GetNativeID() string {
	return fmt.Sprintf("%d", o.forgejoReview.ID)
}

func (o *review) NewFormat() f3.Interface {
	node := o.GetNode()
	return node.GetTree().(f3_tree.TreeInterface).NewFormat(node.GetKind())
}

var sdkStateToFormatState = map[string]string{
	string(forgejo_sdk.ReviewStateApproved):       f3.ReviewStateApproved,
	string(forgejo_sdk.ReviewStatePending):        f3.ReviewStatePending,
	string(forgejo_sdk.ReviewStateComment):        f3.ReviewStateCommented,
	string(forgejo_sdk.ReviewStateRequestChanges): f3.ReviewStateChangesRequested,
	string(forgejo_sdk.ReviewStateRequestReview):  f3.ReviewStateRequestReview,
	string(forgejo_sdk.ReviewStateUnknown):        f3.ReviewStateUnknown,
}

var formatStateToSdkState = func() map[string]string {
	r := make(map[string]string, len(sdkStateToFormatState))
	for k, v := range sdkStateToFormatState {
		r[v] = k
	}
	return r
}()

func convertState(m map[string]string, unknown, fromState string) string {
	if toState, ok := m[fromState]; ok {
		return toState
	}
	return unknown
}

func (o *review) ToFormat() f3.Interface {
	if o.forgejoReview == nil {
		return o.NewFormat()
	}

	review := &f3.Review{
		Common:    f3.NewCommon(o.GetNativeID()),
		Official:  o.forgejoReview.Official,
		CommitID:  o.forgejoReview.CommitID,
		Content:   o.forgejoReview.Body,
		CreatedAt: o.forgejoReview.Submitted,
		State:     convertState(sdkStateToFormatState, f3.ReviewStateUnknown, string(o.forgejoReview.State)),
	}

	if o.forgejoReview.Reviewer != nil {
		review.ReviewerID = f3_tree.NewUserReference(o.forgejoReview.Reviewer.ID)
	}

	return review
}

func (o *review) FromFormat(content f3.Interface) {
	review := content.(*f3.Review)
	o.forgejoReview = &forgejo_sdk.PullReview{
		ID: util.ParseInt(review.GetID()),
		Reviewer: &forgejo_sdk.User{
			ID: review.ReviewerID.GetIDAsInt(),
		},
		Official:  review.Official,
		CommitID:  review.CommitID,
		Body:      review.Content,
		Submitted: review.CreatedAt,
		State:     forgejo_sdk.ReviewStateType(convertState(formatStateToSdkState, string(forgejo_sdk.ReviewStateUnknown), review.State)),
	}
}

func (o *review) Get(ctx context.Context) bool {
	node := o.GetNode()
	o.Trace("%s", node.GetID())

	owner := f3_tree.GetOwnerName(o.GetNode())
	project := f3_tree.GetProjectName(o.GetNode())
	pullRequest := f3_tree.GetPullRequest(o.GetNode())

	review, resp, err := o.getClient().GetPullReview(owner, project, pullRequest.GetID().Int64(), node.GetID().Int64())
	if resp.StatusCode == 404 {
		return false
	}
	if err != nil {
		panic(fmt.Errorf("review %v %w", o, err))
	}
	o.forgejoReview = review
	return true
}

func (o *review) Put(ctx context.Context) id.NodeID {
	node := o.GetNode()
	o.Trace("%s", node.GetID())

	owner := f3_tree.GetOwnerName(o.GetNode())
	project := f3_tree.GetProjectName(o.GetNode())
	pullRequest := f3_tree.GetPullRequest(o.GetNode())

	o.maybeSudoID(ctx, o.forgejoReview.Reviewer.ID)
	defer o.notSudo()

	if o.forgejoReview.Body == "" {
		switch o.forgejoReview.State {
		case forgejo_sdk.ReviewStateRequestReview, forgejo_sdk.ReviewStateRequestChanges, forgejo_sdk.ReviewStateComment:
			o.forgejoReview.Body = "added by F3"
		}
	}

	opt := forgejo_sdk.CreatePullReviewOptions{
		State: o.forgejoReview.State,
		Body:  o.forgejoReview.Body,
	}
	review, resp, err := o.getClient().CreatePullReview(owner, project, pullRequest.GetID().Int64(), opt)
	if err != nil {
		panic(fmt.Errorf("review %s/%s/pulls/%s/reviews %+v: %v", owner, project, pullRequest.GetID(), opt, resp))
	}
	o.forgejoReview = review
	return id.NewNodeID(o.GetNativeID())
}

func (o *review) Patch(ctx context.Context) {
	panic("cannot edit an existing review")
}

func (o *review) Delete(ctx context.Context) {
	node := o.GetNode()
	o.Trace("%s", node.GetID())

	owner := f3_tree.GetOwnerName(o.GetNode())
	project := f3_tree.GetProjectName(o.GetNode())
	pullRequest := f3_tree.GetPullRequest(o.GetNode())

	_, err := o.getClient().DeletePullReview(owner, project, pullRequest.GetID().Int64(), node.GetID().Int64())
	if err != nil {
		panic(err)
	}
}
