#![allow(missing_docs)]

/// Error enum representing different errors that can occur in this crate.
#[derive(Debug, thiserror::Error)]
pub enum Error {
    /// Error downloading crate sources from crates.io.
    #[error(transparent)]
    Download {
        #[from]
        inner: reqwest::Error,
    },

    /// Error reading / writing files to / from disk or /tmp space.
    #[error(transparent)]
    IO {
        #[from]
        inner: std::io::Error,
    },

    /// Error querying crates.io for the specified crate.
    #[error("Could not find crate on crates.io: {name}")]
    CrateNotFound { name: String },

    /// Error querying crates.io for the specified crate version.
    #[error("Could not find crate version on crates.io: {name}@{version}")]
    VersionNotFound { name: String, version: String },

    /// Error parsing the repository URL from crate metadata.
    #[error("Invalid repository URL: {repo}")]
    InvalidRepoUrl { repo: String },

    /// Error checking out the specified git ref from the repository.
    #[error("Invalid git ref for repository: {repo}#{rev}")]
    InvalidGitRef { repo: String, rev: String },

    /// Error caused by failing to find a crate inside a repository
    #[error("Cannot determine crate path in repository: {repo} / {name}")]
    PathNotDeterminable { repo: String, name: String },

    /// Other errors from a subprocess (i.e. git).
    #[error("Process failed: [{cmd}]\n{stdout}\n{stderr}")]
    Subprocess {
        cmd: String,
        stdout: String,
        stderr: String,
    },

    /// Error loading crate metadata.
    #[error("Failed to load crate metadata: {inner}")]
    Metadata { inner: String },

    /// Error walking the source directory of a crate.
    #[error("Failed to walk source directory: {inner}")]
    Walk { inner: String },

    /// Error parsing the `.cargo_vcs_info.json` file.
    #[error("Failed to load '.cargo_vcs_info.json' file: {inner}")]
    VcsInfo { inner: String },
}
