use presentation;
use core:geometry;
use graphics;
use ui;

/**
 * Class representing the axis environment for a horizontal bar graph. This class is basically a
 * layout manager for the individual components of the graph.
 */
class BarGraph extends Element {
	// The axis owned by the graph.
	package GraphAxis axis;

	// Labels in use.
	package GraphLabel[] labels;

	// Data sets to plot.
	package GraphData[] data;

	// Create.
	init(Presentation p) {
		init() {
			axis(p.contentStyle);
		}
	}

	// Set the font used in the axis.
	void axisStyle(TextStyle style) {
		axis.textStyle = style;
	}

	// Set the label style.
	void labelStyle(TextStyle style) {
		axis.labelStyle = style;
	}

	// Set the ticks on the x-axis.
	void ticks(Float min, Float step, Float max) {
		axis.ticks(min, step, max);
	}

	// Set margin on the left.
	void leftMargin(Float margin) {
		axis.marginLeft = margin;
	}

	// Set margin on the right.
	void rightMargin(Float margin) {
		axis.marginRight = margin;
	}

	// Set labels on the y-axis.
	void labels(Str[] labels) {
		this.labels.clear();

		for (l in labels)
			this.labels << GraphLabel(Text(l, axis.textStyle.font));
	}

	// Indent a specific label to the right margin. Used to indicate sub-groups.
	void indentLabel(Nat id) {
		labels[id].indent = axis.marginLeft;
	}

	// Set the indentation color.
	void indentColor(Color color) {
		axis.indentColor = SolidBrush(color);
	}

	// Scale a specific label relative to the default size.
	void scaleLabel(Nat id, Float scale) {
		scaleLabel(id, scale, 1.0);
	}

	void scaleLabel(Nat id, Float heightScale, Float textScale) {
		labels[id].displayScale = heightScale;
		labels[id].textScale = textScale;
	}

	// Margin between individual bars.
	void barMargin(Float v) {
		axis.barMargin = v;
	}

	// Margin between groups of bars.
	void barGroupMargin(Float v) {
		axis.barGroupMargin = v;
	}

	// Maximum height of a single bar.
	void barMaxHeight(Float v) {
		axis.barMaxHeight = v;
	}

	// Add some data.
	void data(Float[] values, Color color) {
		data << GraphData(values, color);
	}

	// Generate labels for the bars by using a lambda function.
	void dataLabels(fn(Float)->Str fn) {
		for (d in data)
			d.labels(fn, axis.labelStyle.font);
	}

	// Get the minimum size of the layout.
	Size minSize() {
		Size(0, 0);
	}

	// Animation handling.
	void draw(Graphics g, Nat step, Duration time) : override {
		for (label in labels)
			label.animationDefault = 0xFFFFFFFF;

		if (before(g, step, time))
			draw(g);
		after(g, step, time);
	}

	// Draw this element.
	void draw(Graphics g) : override {
		Rect graph = pos;
		graph.p1.y -= axis.tickHeight() + axis.labelMargin;

		Float numBars = 0;
		for (label in labels)
			numBars += label.scale;

		Float leftMargin;
		Float height = graph.size.h / numBars;
		Float barHeight = (height - axis.barGroupMargin) / data.count.int.float;
		barHeight = min(barHeight, axis.barMaxHeight);
		height = barHeight * data.count.int.float + axis.barGroupMargin;

		Float y;
		for (label in labels) {
			Float h = height * label.scale;
			label.totalHeight = h;
			label.height = (h - axis.barGroupMargin) / data.count.int.float;
			label.offset = y + h / 2;
			y += h;

			leftMargin = max(leftMargin, label.text.size.w);
		}

		graph.p0.x += leftMargin + axis.labelMargin;;

		// In case we made the graph smaller.
		graph.p1.y = y + graph.p0.y;

		// Compute offsets for each data set.
		for (i, d in data) {
			d.offset = i.int.float - (data.count - 1).int.float / 2;
		}

		// Draw data.
		for (d in data) {
			d.draw(g, graph, labels, axis);
		}

		// Draw the axis itself.
		axis.draw(g, graph, labels);
	}
}

// Helper for creating the bar graph.
BarGraph barGraph(Presentation p) {
	BarGraph(p);
}


/**
 * Class responsible for drawing the axis itself.
 */
class GraphAxis on Render {
	// The style of the text on the axis.
	TextStyle textStyle;

	// Style of labels.
	TextStyle labelStyle;

	// Color of indentations.
	Brush indentColor;

	// Created tick labels.
	private Text[] ticks;

	// Minimum value.
	private Float min;

	// Maximum value.
	private Float max;

	// Step for the ticks.
	private Float step;

	// Text color. Copy so that we can freely modify its opacity!
	private SolidBrush textBrush;

	// Margin on the left side (pixels).
	Float marginLeft;

	// Margin on the right side (pixels).
	Float marginRight;

	// Size of the markers.
	Float markSize;

	// Margin between the axis and the labels.
	Float labelMargin;

	// Margin between groups of bars.
	Float barGroupMargin;

	// Margin between individual bars.
	Float barMargin;

	// Maximum height of a single bar.
	Float barMaxHeight;

	// The arrow we will draw.
	private Path arrow;

	// Create.
	init(TextStyle style) {
		Color textColor = black;
		if (b = style.fill as SolidBrush) {
			textColor = b.color;
			textColor.a *= b.opacity;
		} else {
			print("WARNING: We ignore non-solid text styles for axis labels in a graph!");
		}

		init() {
			textStyle = style;
			labelStyle = style;
			indentColor = SolidBrush(white * 0.5);
			textBrush = SolidBrush(textColor);
			min = 0;
			max = 1;
			step = 1;
			marginLeft = 0;
			marginRight = 20;
			markSize = 4;
			labelMargin = 4;
			barGroupMargin = 10;
			barMargin = 5;
			barMaxHeight = 40;
		}

		ticks(0, 1, 1);
		arrowSize(10);
	}

	// Create ticks for the graph.
	void ticks(Float min, Float step, Float max) {
		ticks.clear();
		this.min = min;
		this.max = max;
		this.step = step;

		Int index = 0;
		do {
			Float at = min + index.float*step;
		} while (at <= max) {
			ticks << Text("${at}", textStyle.font);
			index++;
		}
	}

	// Compute the height of the tick labels.
	Float tickHeight() {
		Float h;
		for (l in ticks)
			h = max(h, l.size.h);
		h;
	}

	// Draw the axis.
	void draw(Graphics g, Rect graph, GraphLabel[] labels) {
		// The Y axis.
		if (marginLeft > 0.1) {
			g.line(graph.p0, Point(graph.p0.x, graph.p1.y), textStyle.fill);
		} else {
			g.line(graph.p0, Point(graph.p0.x, graph.p1.y + markSize / 2), textStyle.fill);
		}

		Point indentFrom(0, graph.p0.y);
		Float indentTo;

		for (l in labels) {
			if (l.scale > 0) {
				Size sz = l.text.size;
				Point align = graph.p0 + Point(0, l.offset);

				g.line(align - Point(markSize / 2, 0), align, textStyle.fill);

				textBrush.opacity = l.animationScale;

				align.x -= labelMargin;
				if (l.textScale == 1.0) {
					align -= Size(sz.w, sz.h / 2);
					g.draw(l.text, textBrush, align);
				} else {
					g.push();
					g.transform(scale(l.textScale) * translate(align));
					g.draw(l.text, textBrush, Point(-sz.w, -sz.h / 2));
					g.pop();
				}

				textBrush.opacity = 1;

				if (indentFrom.x != l.indent) {
					Float y = graph.p0.y + l.offset - l.totalHeight/2;
					if (indentFrom.x > 0) {
						indentFrom.x += graph.p0.x;
						Rect bump = Rect(indentFrom, Point(graph.p0.x, y)).normalized;
						g.fill(bump, indentColor);
						g.draw(bump, textStyle.fill);
					} else {
						indentTo = indentFrom.y + l.totalHeight;
					}

					indentFrom = Point(l.indent, y);
				}
			}
		}

		if (indentFrom.x > 0) {
			indentFrom.x += graph.p0.x;
			Rect bump = Rect(indentFrom, Point(graph.p0.x, indentTo)).normalized;
			g.fill(bump, indentColor);
			g.draw(bump, textStyle.fill);
		}

		// The X axis.
		g.line(Point(graph.p0.x, graph.p1.y), graph.p1, textStyle.fill);

		g.push();
		g.transform(translate(graph.p1));
		g.draw(arrow, textStyle.fill);
		g.pop();

		for (id, text in ticks) {
			Float y = min + step*id.int.float;
			Point p(coordToPx(graph, y), graph.p1.y);

			if (id > 0 | marginLeft > 0.1)
				g.line(p - Point(0, markSize/2), p + Point(0, markSize/2), textStyle.fill);

			Size sz = text.size;
			p -= Size(sz.w / 2, -labelMargin);
			g.draw(text, textStyle.fill, p);
		}
	}

	// Compute the pixel value of a coordinate (x coordinate)
	Float coordToPx(Rect graph, Float y) {
		Float l = graph.p0.x + marginLeft;
		Float r = graph.p1.x - marginRight;
		Float step = (r - l) / (max - min);
		l + step*(y - min);
	}

	// Set the arrow size.
	assign arrowSize(Float size) {
		arrow.clear();

		Float height = size * 2 / 3;
		arrow.start(Point(-size, -height));
		arrow.line(Point(0, 0));
		arrow.line(Point(-size, height));
	}
}

/**
 * Class representing a label in a graph.
 */
class GraphLabel on Render {
	// The title of this label.
	Text text;

	// Y-offset of this label.
	Float offset;

	// Label indentation (i.e. start bars at this x-offset).
	Float indent;

	// Height of each bar in this label.
	Float height;

	// Total height of all bars in this label.
	Float totalHeight;

	// Scale of this label (used to show/hide parts of the graph). Not shown if zero.
	Float scale() {
		animationScale * displayScale;
	}

	// Scaling applied by animations.
	Float animationScale;

	// Scaling applied by config.
	Float displayScale;

	// Text scale for this label (not the text inside the graph).
	Float textScale;

	// The lowest animation ID that has set the default value for 'scale'.
	Nat animationDefault;

	// Create.
	init(Text text) {
		init {
			text = text;
			animationScale = 1.0;
			displayScale = 1.0;
			textScale = 1.0;
		}
	}
}

/**
 * Class representing one set of data-points in a graph.
 */
class GraphData on Render {
	// A single value.
	value Value {
		// The value itself. Used for plotting.
		Float point;

		// Text used as a label (optional).
		Text? label;

		// Create.
		init(Float p) {
			init { point = p; }
		}
	}

	// Data.
	Value[] values;

	// Offset of this graph (relative to the "height").
	Float offset;

	// Color for this data set.
	SolidBrush color;

	// Create the data.
	init(Float[] values, Color color) {
		init() {
			color(color);
		}

		for (v in values)
			this.values << Value(v);
	}

	// Create data labels.
	void labels(fn(Float)->Str fn, Font font) {
		for (Nat i = 0; i < values.count; i++) {
			if (values[i].label) {
			} else {
				values[i].label = Text(fn.call(values[i].point), font);
			}
		}
	}

	// Draw this graph.
	void draw(Graphics g, Rect graph, GraphLabel[] labels, GraphAxis axis) {
		for (i, label in labels) {
			if (i < values.count) {
				drawBar(g, graph, label, values[i], axis);
			}
		}
	}

	// Draw a single bar.
	private void drawBar(Graphics g, Rect graph, GraphLabel label, Value value, GraphAxis axis) {
		Float y = label.offset + graph.p0.y;
		y += offset * label.height;

		Float from = graph.p0.x + label.indent;
		Float to = axis.coordToPx(graph, value.point);

		Float height = (label.height - axis.barMargin) / 2;
		if (height <= 0.0)
			return;

		Rect bar(Point(from, y - height), Point(to, y + height));
		g.draw(bar, axis.labelStyle.fill);
		g.fill(bar, color);

		if (text = value.label) {
			Float scale = label.textScale;

			Size sz = text.size;
			Point anchor(to + axis.labelMargin, y - sz.h * scale / 2);
			if (anchor.x + sz.w * scale > graph.p1.x)
				anchor.x = to - axis.labelMargin - sz.w * scale;

			if (scale == 1) {
				g.draw(text, axis.labelStyle.fill, anchor);
			} else {
				g.push();
				g.transform(scale(scale) * translate(anchor));
				g.draw(text, axis.labelStyle.fill, Point());
				g.pop();
			}
		}
	}
}


/**
 * Animation that shows a particular sub-graph at the given time.
 */
class ShowLabelAnimation extends Animation {
	// The label id we shall work with.
	private Nat labelId;

	init(Nat step, Nat id) {
		init(step) {
			labelId = id;
		}
	}

	Bool before(Element element, Graphics g, Nat cStep, Duration time) : override {
		if (element as BarGraph) {
			var label = element.labels[labelId];

			if (cStep <= step) {
				if (label.animationDefault > step) {
					label.animationScale = 0;
					label.animationDefault = step;
				}
			} else {
				label.animationScale = 1;
			}


			if (cStep == step) {
				label.animationScale = smoothVal(time);
			}
		}

		true;
	}
	void after(Element element, Graphics g, Nat cStep, Duration time) : override {}
}

/**
 * Animation that shows a particular sub-graph at the given time.
 */
class HideLabelAnimation extends Animation {
	// The label id we shall work with.
	private Nat labelId;

	init(Nat step, Nat id) {
		init(step) {
			labelId = id;
		}
	}

	Bool before(Element element, Graphics g, Nat cStep, Duration time) : override {
		if (element as BarGraph) {
			var label = element.labels[labelId];

			if (cStep <= step) {
				if (label.animationDefault > step) {
					label.animationScale = 1;
					label.animationDefault = step;
				}
			} else {
				label.animationScale = 0;
			}

			if (step == cStep) {
				label.animationScale = 1.0 - smoothVal(time);
			}
		}

		true;
	}
	void after(Element element, Graphics g, Nat cStep, Duration time) : override {}
}
