/* -*- Mode: C++; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */
/*
 * This file is part of the LibreOffice project.
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 * This file incorporates work covered by the following license notice:
 *
 *   Licensed to the Apache Software Foundation (ASF) under one or more
 *   contributor license agreements. See the NOTICE file distributed
 *   with this work for additional information regarding copyright
 *   ownership. The ASF licenses this file to you under the Apache
 *   License, Version 2.0 (the "License"); you may not use this file
 *   except in compliance with the License. You may obtain a copy of
 *   the License at http://www.apache.org/licenses/LICENSE-2.0 .
 */

#ifndef INCLUDED_SVX_SVDDRAG_HXX
#define INCLUDED_SVX_SVDDRAG_HXX


#include <tools/gen.hxx>
#include <tools/fract.hxx>
#include <svx/svxdllapi.h>

#include <memory>
#include <vector>

// Status information for specialized object dragging. In order for the model
// to stay status free, the status data is kept on the View
// and handed over to the object at the appropriate time as a parameter.
// This also includes the status of the operation and Interactive
// Object creation. In this case, pHdl is null.
class SdrHdl;
class SdrView;
class SdrPageView;
class SdrDragMethod;

struct SVX_DLLPUBLIC SdrDragStatUserData
{
    virtual ~SdrDragStatUserData() = 0;
};

class SVX_DLLPUBLIC SdrDragStat final
{
    SdrHdl*  pHdl;      // The Handle for the User
    SdrView* pView;
    SdrPageView* pPageView;
    std::vector<Point> mvPnts; // All previous Points: [0]=Start, [Count()-2]=Prev
    Point     aRef1;     // Referencepoint: Resize fixed point, (axis of rotation,
    Point     aRef2;     // axis of reflection, ...)
    Point     aPos0;     // Position at the last Event
    Point     aRealNow;  // Current dragging position without Snap, Ortho and Limit
    tools::Rectangle aActionRect;

    bool      bEndDragChangesAttributes;
    bool      bEndDragChangesGeoAndAttributes;
    /// Table row drag: table will re-layout itself later.
    bool      mbEndDragChangesLayout;
    bool      bMouseIsUp;

    bool      bShown;    // Xor visible?
    sal_uInt16    nMinMov;   // So much has to be minimally moved first
    bool      bMinMoved; // MinMove surpassed?

    bool      bHorFixed; // Dragging only vertical
    bool      bVerFixed; // Dragging only horizontal
    bool      bWantNoSnap; // To decide if pObj-> MovCreate () should use NoSnapPos or not.
                          // Therefore, NoSnapPos is written into the buffer.
    bool  bOrtho4;
    bool  bOrtho8;

    SdrDragMethod* pDragMethod;
    std::unique_ptr<SdrDragStatUserData>  mpUserData;     // Userdata

    void Clear();

    sal_Int32 GetPrevPos() const { return mvPnts.size()-(mvPnts.size()>1 ? 2 : 1); }

public:
    SdrDragStat()                                    { Reset(); }
    ~SdrDragStat();
    void         Reset();
    SdrView*     GetView() const                     { return pView; }
    void         SetView(SdrView* pV)                { pView=pV; }
    SdrPageView* GetPageView() const                 { return pPageView; }
    void         SetPageView(SdrPageView* pPV)       { pPageView=pPV; }
    const Point& GetPoint(sal_Int32 nNum) const      { return mvPnts[nNum]; }
    sal_Int32    GetPointCount() const               { return mvPnts.size(); }
    const Point& GetStart() const                    { return mvPnts[0]; }
    const Point& GetPrev() const                     { return mvPnts[GetPrevPos()]; }
    const Point& GetPos0() const                     { return aPos0;  }
    const Point& GetNow() const                      { return mvPnts.back(); }
    void         SetNow(Point const &pt)             { mvPnts.back() = pt; }
    const Point& GetRef1() const                     { return aRef1;  }
    void         SetRef1(const Point &pt)            { aRef1 = pt;  }
    const Point& GetRef2() const                     { return aRef2;  }
    void         SetRef2(const Point &pt)            { aRef2 = pt;  }
    const        SdrHdl* GetHdl() const              { return pHdl;   }
    void         SetHdl(SdrHdl* pH)                  { pHdl=pH;       }
    SdrDragStatUserData* GetUser() const             { return mpUserData.get();  }
    void         SetUser(std::unique_ptr<SdrDragStatUserData> pU) { mpUserData = std::move(pU); }
    bool         IsShown() const                     { return bShown; }
    void         SetShown(bool bOn)                  { bShown=bOn; }

    bool         IsMinMoved() const                  { return bMinMoved; }
    void         SetMinMoved()                       { bMinMoved=true; }
    void         ResetMinMoved()                     { bMinMoved=false; }
    void         SetMinMove(sal_uInt16 nDist)        { nMinMov=nDist; if (nMinMov<1) nMinMov=1; }

    bool         IsHorFixed() const                  { return bHorFixed; }
    void         SetHorFixed(bool bOn)               { bHorFixed=bOn; }
    bool         IsVerFixed() const                  { return bVerFixed; }
    void         SetVerFixed(bool bOn)               { bVerFixed=bOn; }

    // Here, the object can say: "I do not want to snap to coordinates!"
    // for example, the angle of the arc ...
    bool         IsNoSnap() const                     { return bWantNoSnap; }
    void         SetNoSnap(bool bOn = true)           { bWantNoSnap=bOn; }

    // And here the Obj say which Ortho (if there is one) can be usefully applied to him.
    // Ortho4 means Ortho in four directions (for Rect and CIRT)
    bool         IsOrtho4Possible() const             { return bOrtho4; }
    void         SetOrtho4Possible(bool bOn = true)   { bOrtho4=bOn; }
    // Ortho8 means Ortho in 8 directions (for lines)
    bool         IsOrtho8Possible() const             { return bOrtho8; }
    void         SetOrtho8Possible(bool bOn = true)   { bOrtho8=bOn; }

    // Is set by an object that was dragged.
    bool         IsEndDragChangesAttributes() const    { return bEndDragChangesAttributes; }
    void         SetEndDragChangesAttributes(bool bOn) { bEndDragChangesAttributes=bOn; }
    bool         IsEndDragChangesGeoAndAttributes() const   { return bEndDragChangesGeoAndAttributes; }
    void         SetEndDragChangesGeoAndAttributes(bool bOn) { bEndDragChangesGeoAndAttributes=bOn; }
    bool         IsEndDragChangesLayout() const   { return mbEndDragChangesLayout; }
    void         SetEndDragChangesLayout(bool bOn) { mbEndDragChangesLayout=bOn; }

    // Is set by the view and can be evaluated by Obj
    bool         IsMouseDown() const                  { return !bMouseIsUp; }
    void         SetMouseDown(bool bDown)         { bMouseIsUp=!bDown; }

    void         Reset(const Point& rPnt);
    void         NextMove(const Point& rPnt);
    void         NextPoint();
    void         PrevPoint();
    bool         CheckMinMoved(const Point& rPnt);
    long         GetDX() const                     { return GetNow().X()-GetPrev().X(); }
    long         GetDY() const                     { return GetNow().Y()-GetPrev().Y(); }
    Fraction     GetXFact() const;
    Fraction     GetYFact() const;

    SdrDragMethod* GetDragMethod() const             { return pDragMethod; }
    void         SetDragMethod(SdrDragMethod* pMth)  { pDragMethod=pMth; }

    const tools::Rectangle& GetActionRect() const          { return aActionRect; }
    void         SetActionRect(const tools::Rectangle& rR) { aActionRect=rR; }

    // Also considering 1stPointAsCenter
    void         TakeCreateRect(tools::Rectangle& rRect) const;
};

#endif // INCLUDED_SVX_SVDDRAG_HXX

/* vim:set shiftwidth=4 softtabstop=4 expandtab: */
