/* -*- Mode: C++; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */
/*
 * This file is part of the LibreOffice project.
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 * This file incorporates work covered by the following license notice:
 *
 *   Licensed to the Apache Software Foundation (ASF) under one or more
 *   contributor license agreements. See the NOTICE file distributed
 *   with this work for additional information regarding copyright
 *   ownership. The ASF licenses this file to you under the Apache
 *   License, Version 2.0 (the "License"); you may not use this file
 *   except in compliance with the License. You may obtain a copy of
 *   the License at http://www.apache.org/licenses/LICENSE-2.0 .
 */

#ifndef INCLUDED_VCL_OPENGLGDIIMPL_HXX
#define INCLUDED_VCL_OPENGLGDIIMPL_HXX

#include <vcl/dllapi.h>
#include <vcl/opengl/OpenGLContext.hxx>

#include <regionband.hxx>
#include <salgeom.hxx>
#include <salgdiimpl.hxx>
#include <opengl/program.hxx>
#include <opengl/texture.hxx>
#include <opengl/RenderList.hxx>

#include <memory>

class SalFrame;
class SalVirtualDevice;
class OpenGLTests;

namespace basegfx
{
class B2DTrapezoid;
};

namespace tools
{
    class Polygon;
    class PolyPolygon;
}

struct TextureCombo
{
    std::unique_ptr<OpenGLTexture> mpTexture;
    std::unique_ptr<OpenGLTexture> mpMask;
};

class OpenGLFlushIdle;

class VCL_DLLPUBLIC OpenGLSalGraphicsImpl : public SalGraphicsImpl
{
    friend class OpenGLTests;
protected:

    /// This context is solely for blitting maOffscreenTex
    rtl::Reference<OpenGLContext> mpWindowContext;

    /// This context is whatever is most convenient to render
    /// to maOffscreenTex with.
    rtl::Reference<OpenGLContext> mpContext;

    SalGraphics& mrParent;
    /// Pointer to the SalFrame or SalVirtualDevice
    SalGeometryProvider* mpProvider;
    OpenGLProgram* mpProgram;

    /// This idle handler is used to swap buffers after rendering.
    std::unique_ptr<OpenGLFlushIdle> mpFlush;

    // clipping
    vcl::Region maClipRegion;
    bool mbUseScissor;
    bool mbUseStencil;

    bool mbXORMode;

    bool mbAcquiringOpenGLContext;

    /**
     * All rendering happens to this off-screen texture. For
     * non-virtual devices, ie. windows - we will blit it and
     * swapBuffers later.
     */
    OpenGLTexture maOffscreenTex;

    Color mnLineColor;
    Color mnFillColor;
#ifdef DBG_UTIL
    bool mProgramIsSolidColor;
#endif
    sal_uInt32 mnDrawCount;
    sal_uInt32 mnDrawCountAtFlush;
    Color mProgramSolidColor;
    double mProgramSolidTransparency;

    std::unique_ptr<RenderList> mpRenderList;

    void ImplInitClipRegion();
    void ImplSetClipBit( const vcl::Region& rClip, GLuint nMask );
    void ImplDrawLineAA( double nX1, double nY1, double nX2, double nY2, bool edge );
    void CheckOffscreenTexture();

    void ApplyProgramMatrices(float fPixelOffset = 0.0);

public:
    bool UseProgram( const OUString& rVertexShader, const OUString& rFragmentShader, const OString& preamble = "" );
    bool UseSolid( Color nColor, sal_uInt8 nTransparency );
    bool UseSolid( Color nColor, double fTransparency );
    bool UseSolid( Color nColor );
    void UseSolid();
    bool UseLine(Color nColor, double fTransparency, GLfloat fLineWidth, bool bUseAA);
    void UseLine(GLfloat fLineWidth, bool bUseAA);
    bool UseInvert50();
    bool UseInvert(SalInvert nFlags);

    void DrawConvexPolygon( sal_uInt32 nPoints, const Point* pPtAry, bool blockAA = false );
    void DrawConvexPolygon( const tools::Polygon& rPolygon, bool blockAA );
    void DrawTrapezoid( const basegfx::B2DTrapezoid& trapezoid, bool blockAA );
    void DrawRect( tools::Long nX, tools::Long nY, tools::Long nWidth, tools::Long nHeight );
    void DrawRect( const tools::Rectangle& rRect );
    void DrawPolygon( sal_uInt32 nPoints, const Point* pPtAry );
    void DrawLineSegment(float x1, float y1, float x2, float y2);
    void DrawPolyPolygon( const basegfx::B2DPolyPolygon& rPolyPolygon, bool blockAA = false );
    void DrawRegionBand( const RegionBand& rRegion );
    void DrawTextureRect( const SalTwoRect& rPosAry );
    void DrawTexture( OpenGLTexture& rTexture, const SalTwoRect& rPosAry, bool bInverted = false );
    void DrawTransformedTexture( OpenGLTexture& rTexture, OpenGLTexture& rMask, const basegfx::B2DPoint& rNull, const basegfx::B2DPoint& rX, const basegfx::B2DPoint& rY );
    void DrawAlphaTexture( OpenGLTexture& rTexture, const SalTwoRect& rPosAry, bool bInverted, bool pPremultiplied );
    void DrawTextureDiff( OpenGLTexture& rTexture, OpenGLTexture& rMask, const SalTwoRect& rPosAry, bool bInverted );
    void DrawTextureWithMask( OpenGLTexture& rTexture, OpenGLTexture& rMask, const SalTwoRect& rPosAry );
    void DrawBlendedTexture( OpenGLTexture& rTexture, OpenGLTexture& rMask, OpenGLTexture& rAlpha, const SalTwoRect& rPosAry );
    void DrawMask( OpenGLTexture& rTexture, Color nMaskColor, const SalTwoRect& rPosAry );
    void DrawLinearGradient( const Gradient& rGradient, const tools::Rectangle& rRect );
    void DrawAxialGradient( const Gradient& rGradient, const tools::Rectangle& rRect );
    void DrawRadialGradient( const Gradient& rGradient, const tools::Rectangle& rRect );

    void FlushDeferredDrawing();
    void FlushLinesOrTriangles(DrawShaderType eType, RenderParameters const & rParameters);

public:
    // get the width of the device
    GLfloat GetWidth() const { return mpProvider ? mpProvider->GetWidth() : 1; }

    // get the height of the device
    GLfloat GetHeight() const { return mpProvider ? mpProvider->GetHeight() : 1; }

    /**
     * check whether this instance is used for offscreen (Virtual Device)
     * rendering ie. does it need its own context.
     */
    bool IsOffscreen() const { return mpProvider == nullptr || mpProvider->IsOffScreen(); }

    /// Oddly not all operations obey the XOR option.
    enum XOROption { IGNORE_XOR, IMPLEMENT_XOR };

    // initialize pre-draw state
    void InitializePreDrawState(XOROption eOpt);

    // operations to do before painting
    void PreDraw(XOROption eOpt = IGNORE_XOR);

    // operations to do after painting
    void PostDraw();

    void PostBatchDraw();

protected:
    bool AcquireContext(bool bForceCreate = false);
    void ReleaseContext();

    /// create a new context for rendering to the underlying window
    virtual rtl::Reference<OpenGLContext> CreateWinContext() = 0;

    /// check whether the given context can be used for off-screen rendering
    static bool UseContext( const rtl::Reference<OpenGLContext> &pContext )
    {
        return pContext->isInitialized() &&  // not released by the OS etc.
               pContext->isVCLOnly();
    }

public:
    OpenGLSalGraphicsImpl(SalGraphics& pParent, SalGeometryProvider *pProvider);
    virtual ~OpenGLSalGraphicsImpl () override;

    rtl::Reference<OpenGLContext> GetOpenGLContext();

    virtual void Init() override;

    virtual void DeInit() override;

    virtual void freeResources() override;

    virtual OUString getRenderBackendName() const override { return "opengl"; }

    const vcl::Region& getClipRegion() const;
    virtual bool setClipRegion( const vcl::Region& ) override;

    //
    // get the depth of the device
    virtual sal_uInt16 GetBitCount() const override;

    // get the width of the device
    virtual tools::Long GetGraphicsWidth() const override;

    // set the clip region to empty
    virtual void ResetClipRegion() override;

    // set the line color to transparent (= don't draw lines)

    virtual void SetLineColor() override;

    // set the line color to a specific color
    virtual void SetLineColor( Color nColor ) override;

    // set the fill color to transparent (= don't fill)
    virtual void SetFillColor() override;

    // set the fill color to a specific color, shapes will be
    // filled accordingly
    virtual void SetFillColor( Color nColor ) override;

    // enable/disable XOR drawing
    virtual void SetXORMode( bool bSet, bool bInvertOnly ) override;

    // set line color for raster operations
    virtual void SetROPLineColor( SalROPColor nROPColor ) override;

    // set fill color for raster operations
    virtual void SetROPFillColor( SalROPColor nROPColor ) override;

    // draw --> LineColor and FillColor and RasterOp and ClipRegion
    virtual void drawPixel( tools::Long nX, tools::Long nY ) override;
    virtual void drawPixel( tools::Long nX, tools::Long nY, Color nColor ) override;

    virtual void drawLine( tools::Long nX1, tools::Long nY1, tools::Long nX2, tools::Long nY2 ) override;

    virtual void drawRect( tools::Long nX, tools::Long nY, tools::Long nWidth, tools::Long nHeight ) override;

    virtual void drawPolyLine( sal_uInt32 nPoints, const Point* pPtAry ) override;

    virtual void drawPolygon( sal_uInt32 nPoints, const Point* pPtAry ) override;

    virtual void drawPolyPolygon( sal_uInt32 nPoly, const sal_uInt32* pPoints, const Point** pPtAry ) override;

    virtual bool drawPolyPolygon(
                const basegfx::B2DHomMatrix& rObjectToDevice,
                const basegfx::B2DPolyPolygon&,
                double fTransparency) override;

    virtual bool drawPolyLine(
                const basegfx::B2DHomMatrix& rObjectToDevice,
                const basegfx::B2DPolygon&,
                double fTransparency,
                double fLineWidth,
                const std::vector< double >* pStroke, // MM01
                basegfx::B2DLineJoin,
                css::drawing::LineCap,
                double fMiterMinimumAngle,
                bool bPixelSnapHairline) override;

    virtual bool drawPolyLineBezier(
                sal_uInt32 nPoints,
                const Point* pPtAry,
                const PolyFlags* pFlgAry ) override;

    virtual bool drawPolygonBezier(
                sal_uInt32 nPoints,
                const Point* pPtAry,
                const PolyFlags* pFlgAry ) override;

    virtual bool drawPolyPolygonBezier(
                sal_uInt32 nPoly,
                const sal_uInt32* pPoints,
                const Point* const* pPtAry,
                const PolyFlags* const* pFlgAry ) override;

    // CopyArea --> No RasterOp, but ClipRegion
    virtual void copyArea(
                tools::Long nDestX, tools::Long nDestY,
                tools::Long nSrcX, tools::Long nSrcY,
                tools::Long nSrcWidth, tools::Long nSrcHeight,
                bool bWindowInvalidate ) override;

    // CopyBits and DrawBitmap --> RasterOp and ClipRegion
    // CopyBits() --> pSrcGraphics == NULL, then CopyBits on same Graphics
    void DoCopyBits(const SalTwoRect& rPosAry, OpenGLSalGraphicsImpl &rSrcImpl);

    virtual bool blendBitmap(
                const SalTwoRect&,
                const SalBitmap& rBitmap ) override;

    virtual bool blendAlphaBitmap(
                const SalTwoRect&,
                const SalBitmap& rSrcBitmap,
                const SalBitmap& rMaskBitmap,
                const SalBitmap& rAlphaBitmap ) override;

    virtual void drawBitmap( const SalTwoRect& rPosAry, const SalBitmap& rSalBitmap ) override;

    virtual void drawBitmap(
                const SalTwoRect& rPosAry,
                const SalBitmap& rSalBitmap,
                const SalBitmap& rMaskBitmap ) override;

    virtual void drawMask(
                const SalTwoRect& rPosAry,
                const SalBitmap& rSalBitmap,
                Color nMaskColor ) override;

    virtual std::shared_ptr<SalBitmap> getBitmap( tools::Long nX, tools::Long nY, tools::Long nWidth, tools::Long nHeight ) override;

    virtual Color getPixel( tools::Long nX, tools::Long nY ) override;

    // invert --> ClipRegion (only Windows or VirDevs)
    virtual void invert(
                tools::Long nX, tools::Long nY,
                tools::Long nWidth, tools::Long nHeight,
                SalInvert nFlags) override;

    virtual void invert( sal_uInt32 nPoints, const Point* pPtAry, SalInvert nFlags ) override;

    virtual bool drawEPS(
                tools::Long nX, tools::Long nY,
                tools::Long nWidth, tools::Long nHeight,
                void* pPtr,
                sal_uInt32 nSize ) override;

    /** Render bitmap with alpha channel

        @param rSourceBitmap
        Source bitmap to blit

        @param rAlphaBitmap
        Alpha channel to use for blitting

        @return true, if the operation succeeded, and false
        otherwise. In this case, clients should try to emulate alpha
        compositing themselves
     */
    virtual bool drawAlphaBitmap(
                const SalTwoRect&,
                const SalBitmap& rSourceBitmap,
                const SalBitmap& rAlphaBitmap ) override;

    /** draw transformed bitmap (maybe with alpha) where Null, X, Y define the coordinate system */
    virtual bool drawTransformedBitmap(
                const basegfx::B2DPoint& rNull,
                const basegfx::B2DPoint& rX,
                const basegfx::B2DPoint& rY,
                const SalBitmap& rSourceBitmap,
                const SalBitmap* pAlphaBitmap) override;

    /** Render solid rectangle with given transparency

      @param nX             Top left coordinate of rectangle

      @param nY             Bottom right coordinate of rectangle

      @param nWidth         Width of rectangle

      @param nHeight        Height of rectangle

      @param nTransparency  Transparency value (0-255) to use. 0 blits and opaque, 255 a
                            fully transparent rectangle

      @returns true if successfully drawn, false if not able to draw rectangle
     */
    virtual bool drawAlphaRect(
                    tools::Long nX, tools::Long nY,
                    tools::Long nWidth, tools::Long nHeight,
                    sal_uInt8 nTransparency ) override;

    virtual bool drawGradient(const tools::PolyPolygon& rPolygon, const Gradient& rGradient) override;
    virtual bool implDrawGradient(basegfx::B2DPolyPolygon const & rPolyPolygon, SalGradient const & rGradient) override;

    virtual bool supportsOperation(OutDevSupportType eType) const override;

    /// queue an idle flush of contents of the back-buffer to the screen
    void flush();

public:
    /// do flush of contents of the back-buffer to the screen & swap.
    void doFlush();
};

#endif

/* vim:set shiftwidth=4 softtabstop=4 expandtab: */
